<?php

namespace App\Services;

use App\Models\Tournament;
use App\Models\Bracket;
use App\Models\GameMatch;
use App\Models\Team;
use App\Models\Player;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Facades\Log;

class TournamentAlgorithmTester
{
    protected TournamentBracketService $bracketService;

    public function __construct()
    {
        $this->bracketService = new TournamentBracketService(new Tournament());
    }

    /**
     * Test all tournament algorithms with sample data
     */
    public function testAllAlgorithms(): array
    {
        $results = [];

        // Test single elimination
        $results['single_elimination'] = $this->testSingleElimination();

        // Test double elimination
        $results['double_elimination'] = $this->testDoubleElimination();

        // Test round robin
        $results['round_robin'] = $this->testRoundRobin();

        // Test group knockout
        $results['group_knockout'] = $this->testGroupKnockout();

        // Test Swiss system
        $results['swiss_system'] = $this->testSwissSystem();

        return $results;
    }

    /**
     * Test single elimination algorithm
     */
    public function testSingleElimination(): array
    {
        $participants = $this->generateTestParticipants(16);
        
        $tournament = Tournament::create([
            'name' => 'Test Single Elimination Tournament',
            'type' => 'knockout',
            'format' => 'team',
            'status' => 'active',
            'start_date' => now(),
            'end_date' => now()->addDays(3),
            'bracket_settings' => [
                'seeding_method' => 'balanced',
            ],
        ]);

        $bracketService = new TournamentBracketService($tournament);
        $bracket = $bracketService->generateBracket($participants, 'winners');

        $matches = $bracket->gameMatches;
        $rounds = $matches->groupBy('round');

        return [
            'success' => true,
            'participants' => count($participants),
            'total_matches' => $matches->count(),
            'rounds' => $rounds->count(),
            'matches_per_round' => $rounds->map->count()->toArray(),
            'structure' => $bracket->structure,
            'seeding' => $bracket->seeding,
        ];
    }

    /**
     * Test double elimination algorithm
     */
    public function testDoubleElimination(): array
    {
        $participants = $this->generateTestParticipants(8);
        
        $tournament = Tournament::create([
            'name' => 'Test Double Elimination Tournament',
            'type' => 'double-elimination',
            'format' => 'team',
            'status' => 'active',
            'start_date' => now(),
            'end_date' => now()->addDays(5),
            'bracket_settings' => [
                'seeding_method' => 'rating',
            ],
        ]);

        $bracketService = new TournamentBracketService($tournament);
        $bracket = $bracketService->generateBracket($participants, 'winners');

        $matches = $bracket->gameMatches;
        $winnersMatches = $matches->where('bracket_type', '!=', 'losers');
        $losersMatches = $matches->where('bracket_type', 'losers');

        return [
            'success' => true,
            'participants' => count($participants),
            'total_matches' => $matches->count(),
            'winners_bracket_matches' => $winnersMatches->count(),
            'losers_bracket_matches' => $losersMatches->count(),
            'structure' => $bracket->structure,
        ];
    }

    /**
     * Test round robin algorithm
     */
    public function testRoundRobin(): array
    {
        $participants = $this->generateTestParticipants(6);
        
        $tournament = Tournament::create([
            'name' => 'Test Round Robin Tournament',
            'type' => 'round-robin',
            'format' => 'team',
            'status' => 'active',
            'start_date' => now(),
            'end_date' => now()->addDays(7),
            'bracket_settings' => [
                'seeding_method' => 'random',
            ],
        ]);

        $bracketService = new TournamentBracketService($tournament);
        $bracket = $bracketService->generateBracket($participants, 'winners');

        $matches = $bracket->gameMatches;
        $rounds = $matches->groupBy('round');

        return [
            'success' => true,
            'participants' => count($participants),
            'total_matches' => $matches->count(),
            'rounds' => $rounds->count(),
            'matches_per_round' => $rounds->map->count()->toArray(),
            'structure' => $bracket->structure,
        ];
    }

    /**
     * Test group knockout algorithm
     */
    public function testGroupKnockout(): array
    {
        $participants = $this->generateTestParticipants(12);
        
        $tournament = Tournament::create([
            'name' => 'Test Group Knockout Tournament',
            'type' => 'group-knockout',
            'format' => 'team',
            'status' => 'active',
            'start_date' => now(),
            'end_date' => now()->addDays(10),
            'bracket_settings' => [
                'seeding_method' => 'ranking',
                'groups' => 3,
            ],
        ]);

        $bracketService = new TournamentBracketService($tournament);
        $bracket = $bracketService->generateBracket($participants, 'winners');

        $matches = $bracket->gameMatches;
        $groupMatches = $matches->whereNotNull('group_number');

        return [
            'success' => true,
            'participants' => count($participants),
            'total_matches' => $matches->count(),
            'group_matches' => $groupMatches->count(),
            'groups' => $bracket->structure['groups'],
            'structure' => $bracket->structure,
        ];
    }

    /**
     * Test Swiss system algorithm
     */
    public function testSwissSystem(): array
    {
        $participants = $this->generateTestParticipants(8);
        
        $tournament = Tournament::create([
            'name' => 'Test Swiss System Tournament',
            'type' => 'swiss_system',
            'format' => 'team',
            'status' => 'active',
            'start_date' => now(),
            'end_date' => now()->addDays(6),
            'bracket_settings' => [
                'seeding_method' => 'rating',
                'swiss_rounds' => 4,
            ],
        ]);

        $bracketService = new TournamentBracketService($tournament);
        $bracket = $bracketService->generateBracket($participants, 'winners');

        $matches = $bracket->gameMatches;
        $rounds = $matches->groupBy('round');

        return [
            'success' => true,
            'participants' => count($participants),
            'total_matches' => $matches->count(),
            'rounds' => $rounds->count(),
            'matches_per_round' => $rounds->map->count()->toArray(),
            'structure' => $bracket->structure,
        ];
    }

    /**
     * Test seeding algorithms
     */
    public function testSeedingAlgorithms(): array
    {
        $participants = $this->generateTestParticipants(16);
        $results = [];

        $seedingMethods = ['random', 'ranking', 'rating', 'balanced'];

        foreach ($seedingMethods as $method) {
            $tournament = Tournament::create([
                'name' => "Test Seeding - {$method}",
                'type' => 'knockout',
                'format' => 'team',
                'status' => 'active',
                'start_date' => now(),
                'end_date' => now()->addDays(3),
                'bracket_settings' => [
                    'seeding_method' => $method,
                ],
            ]);

            $bracketService = new TournamentBracketService($tournament);
            $bracket = $bracketService->generateBracket($participants, 'winners');

            $results[$method] = [
                'success' => true,
                'seeding_order' => array_column($bracket->seeding['participants'], 'id'),
                'structure' => $bracket->structure,
            ];
        }

        return $results;
    }

    /**
     * Test bracket advancement
     */
    public function testBracketAdvancement(): array
    {
        $participants = $this->generateTestParticipants(8);
        
        $tournament = Tournament::create([
            'name' => 'Test Bracket Advancement',
            'type' => 'knockout',
            'format' => 'team',
            'status' => 'active',
            'start_date' => now(),
            'end_date' => now()->addDays(3),
        ]);

        $bracketService = new TournamentBracketService($tournament);
        $bracket = $bracketService->generateBracket($participants, 'winners');

        // Simulate completing first round matches
        $firstRoundMatches = $bracket->gameMatches->where('round', 1);
        $winners = [];

        foreach ($firstRoundMatches as $index => $match) {
            // Simulate winner (alternating pattern)
            $winnerId = $index % 2 === 0 ? $match->team1_id : $match->team2_id;
            
            $match->update([
                'status' => 'completed',
                'winning_team_id' => $winnerId,
                'team1_score' => $index % 2 === 0 ? 2 : 1,
                'team2_score' => $index % 2 === 0 ? 1 : 2,
                'completed_at' => now(),
            ]);

            $winners[] = ['id' => $winnerId, 'type' => 'team'];
        }

        // Advance to next round
        $result = $bracketService->advanceRound($bracket, 1);

        return [
            'success' => true,
            'first_round_winners' => $winners,
            'next_round_matches' => $result['matches_created'],
            'next_round' => $result['round'],
        ];
    }

    /**
     * Test tournament with different participant counts
     */
    public function testParticipantCounts(): array
    {
        $results = [];
        $participantCounts = [4, 8, 16, 32, 64];

        foreach ($participantCounts as $count) {
            $participants = $this->generateTestParticipants($count);
            
            $tournament = Tournament::create([
                'name' => "Test Tournament - {$count} participants",
                'type' => 'knockout',
                'format' => 'team',
                'status' => 'active',
                'start_date' => now(),
                'end_date' => now()->addDays(3),
            ]);

            $bracketService = new TournamentBracketService($tournament);
            $bracket = $bracketService->generateBracket($participants, 'winners');

            $results[$count] = [
                'success' => true,
                'participants' => $count,
                'total_matches' => $bracket->gameMatches->count(),
                'rounds' => $bracket->structure['rounds'],
                'byes' => $bracket->structure['byes'] ?? 0,
                'power_of_two' => $bracket->structure['power_of_two'] ?? 0,
            ];
        }

        return $results;
    }

    /**
     * Performance test for large tournaments
     */
    public function testPerformance(): array
    {
        $results = [];
        $largeCounts = [128, 256, 512];

        foreach ($largeCounts as $count) {
            $startTime = microtime(true);
            
            $participants = $this->generateTestParticipants($count);
            
            $tournament = Tournament::create([
                'name' => "Performance Test - {$count} participants",
                'type' => 'knockout',
                'format' => 'team',
                'status' => 'active',
                'start_date' => now(),
                'end_date' => now()->addDays(7),
            ]);

            $bracketService = new TournamentBracketService($tournament);
            $bracket = $bracketService->generateBracket($participants, 'winners');

            $endTime = microtime(true);
            $executionTime = round(($endTime - $startTime) * 1000, 2); // Convert to milliseconds

            $results[$count] = [
                'success' => true,
                'participants' => $count,
                'execution_time_ms' => $executionTime,
                'total_matches' => $bracket->gameMatches->count(),
                'memory_usage_mb' => round(memory_get_peak_usage() / 1024 / 1024, 2),
            ];
        }

        return $results;
    }

    /**
     * Generate test participants with realistic data
     */
    protected function generateTestParticipants(int $count): array
    {
        $participants = [];
        
        // Get or create a test user
        $testUser = \App\Models\User::first();
        if (!$testUser) {
            $testUser = \App\Models\User::create([
                'name' => 'Test User',
                'email' => 'test@example.com',
                'password' => bcrypt('password'),
            ]);
        }
        
        // Create real teams for testing
        for ($i = 1; $i <= $count; $i++) {
            $team = Team::create([
                'name' => "Test Team {$i}",
                'description' => "Test team for algorithm testing",
                'status' => 'active',
            ]);
            
            $participants[] = [
                'id' => $team->id,
                'name' => $team->name,
                'type' => 'team',
                'rating' => rand(800, 2000),
                'ranking' => $i,
                'score' => 0,
            ];
        }

        return $participants;
    }

    /**
     * Clean up test data
     */
    public function cleanupTestData(): void
    {
        DB::table('matches')->where('tournament_id', 'like', '%Test%')->delete();
        DB::table('brackets')->where('tournament_id', 'like', '%Test%')->delete();
        DB::table('tournaments')->where('name', 'like', '%Test%')->delete();
        DB::table('teams')->where('name', 'like', '%Test Team%')->delete();
    }

    /**
     * Run comprehensive test suite
     */
    public function runTestSuite(): array
    {
        Log::info('Starting tournament algorithm test suite');

        $results = [
            'algorithms' => $this->testAllAlgorithms(),
            'seeding' => $this->testSeedingAlgorithms(),
            'advancement' => $this->testBracketAdvancement(),
            'participant_counts' => $this->testParticipantCounts(),
            'performance' => $this->testPerformance(),
        ];

        // Clean up test data
        $this->cleanupTestData();

        Log::info('Tournament algorithm test suite completed', $results);

        return $results;
    }
}
