<?php

namespace App\Services;

use App\Models\Team;
use App\Models\Tournament;
use App\Models\User;
use App\Models\Player;
use App\Models\TeamOfficial;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Facades\Hash;
use Illuminate\Support\Str;

class TeamRegistrationService
{
    /**
     * Register a team for a tournament
     */
    public function registerTeamForTournament(array $data, Tournament $tournament): array
    {
        DB::beginTransaction();
        
        try {
            // Check if team already exists
            $existingTeam = Team::where('name', $data['team']['name'])->first();
            
            if ($existingTeam) {
                // Check if team is already registered for this tournament
                if ($existingTeam->isRegisteredForTournament($tournament)) {
                    throw new \Exception('This team is already registered for this tournament');
                }
                
                $team = $existingTeam;
                $isNewTeam = false;
            } else {
                // Create new team
                $team = Team::create([
                    'name' => $data['team']['name'],
                    'description' => "Team {$data['team']['name']} - Founded {$data['team']['year']}",
                    'captain_id' => null, // Will be set after captain creation
                    'status' => 'active',
                ]);
                $isNewTeam = true;
            }

            // Handle captain creation/selection
            $captain = $this->handleCaptain($data['captain'], $team);

            // Register team for tournament
            $team->registerForTournament($tournament, [
                'notes' => "Registered via public form",
                'tournament_data' => [
                    'registration_method' => 'public_form',
                    'team_email' => $data['team']['email'],
                    'team_phone' => $data['team']['phone'],
                    'team_location' => $data['team']['location'],
                    'team_coach' => $data['team']['coach'],
                    'team_year' => $data['team']['year'],
                ]
            ]);

            // Create players for this tournament
            $players = $this->createPlayers($data['players'], $team, $tournament, $captain);

            // Create team officials for this tournament
            $officials = $this->createOfficials($data['officials'], $team);

            DB::commit();

            return [
                'success' => true,
                'team' => $team,
                'captain' => $captain,
                'players' => $players,
                'officials' => $officials,
                'is_new_team' => $isNewTeam,
            ];

        } catch (\Exception $e) {
            DB::rollback();
            throw $e;
        }
    }

    /**
     * Handle captain creation or selection
     */
    private function handleCaptain(array $captainData, Team $team): User
    {
        // Check if captain already exists
        $existingCaptain = User::where('email', $captainData['email'])->first();
        
        if ($existingCaptain) {
            $captain = $existingCaptain;
        } else {
            // Set temporary registration data for the observer
            // \App\Observers\UserObserver::setTempRegistrationData($captainData['password'], $captainData['name']);
            
            // Create new captain
            $captain = User::create([
                'name' => $captainData['name'],
                'email' => $captainData['email'],
                'phone' => $captainData['phone'],
                'password' => Hash::make($captainData['password']),
                'user_type' => 'team',
                'profile_complete' => false,
                'payment_status' => 'pending',
                'must_change_password' => true,
                'terms_accepted_at' => now(),
            ]);
        }

        // Update team captain if it's a new team
        if (!$team->captain_id) {
            $team->update(['captain_id' => $captain->id]);
        }

        return $captain;
    }

    /**
     * Create players for the tournament
     */
    private function createPlayers(array $playersData, Team $team, Tournament $tournament, User $captain): array
    {
        $players = [];
        $captainPlayerCreated = false;

        foreach ($playersData as $index => $playerData) {
            $isCaptain = $index === 0; // First player is captain
            
            if ($isCaptain && !$captainPlayerCreated) {
                // Create captain's player record for this tournament
                $player = $captain->players()->create([
                    'tournament_id' => $tournament->id,
                    'team_id' => $team->id,
                    'display_name' => $playerData['name'],
                    'phone' => $playerData['phone'],
                    'football_position' => $playerData['position'],
                    'jersey_number' => $playerData['jersey_number'],
                    'is_team_captain' => true,
                    'player_type' => 'amateur',
                    'experience_level' => 'intermediate',
                    'status' => 'registered',
                    'verification_status' => 'pending',
                    'otp_verified' => false,
                ]);
                $captainPlayerCreated = true;
            } else {
                // Check if player user already exists
                $existingPlayerUser = User::where('email', $playerData['email'])->first();
                
                if ($existingPlayerUser) {
                    $playerUser = $existingPlayerUser;
                } else {
                    // Generate temporary password for the player
                    $tempPassword = Str::random(12);
                    
                    // Set temporary registration data for the observer
                    // \App\Observers\UserObserver::setTempRegistrationData($tempPassword, $playerData['name']);
                    
                    // Create a temporary user account for the player
                    $playerUser = User::create([
                        'name' => $playerData['name'],
                        'email' => $playerData['email'],
                        'phone' => $playerData['phone'],
                        'password' => Hash::make($tempPassword), // Temporary password
                        'user_type' => 'player',
                        'profile_complete' => false,
                        'payment_status' => 'pending',
                        'must_change_password' => true,
                        'terms_accepted_at' => now(),
                    ]);
                }
                
                // Create player record for this tournament
                $player = $playerUser->players()->create([
                    'tournament_id' => $tournament->id,
                    'team_id' => $team->id,
                    'display_name' => $playerData['name'],
                    'phone' => $playerData['phone'],
                    'football_position' => $playerData['position'],
                    'jersey_number' => $playerData['jersey_number'],
                    'is_team_captain' => false,
                    'player_type' => 'amateur',
                    'experience_level' => 'beginner',
                    'status' => 'registered',
                    'verification_status' => 'pending',
                    'otp_verified' => false,
                ]);
            }

            $players[] = $player;
        }

        return $players;
    }

    /**
     * Create team officials
     */
    private function createOfficials(array $officialsData, Team $team): array
    {
        $officials = [];

        foreach ($officialsData as $officialData) {
            $official = TeamOfficial::create([
                'team_id' => $team->id,
                'name' => $officialData['name'],
                'email' => $officialData['email'],
                'phone' => $officialData['phone'],
                'role' => $officialData['role'],
                'qualifications' => $officialData['qualifications'] ?? null,
                'otp_verified' => false,
            ]);

            $officials[] = $official;
        }

        return $officials;
    }

    /**
     * Check if a team can register for a tournament
     */
    public function canTeamRegisterForTournament(Team $team, Tournament $tournament): array
    {
        $errors = [];

        // Check if team is already registered
        if ($team->isRegisteredForTournament($tournament)) {
            $errors[] = 'This team is already registered for this tournament';
        }

        // Check if tournament registration is open
        if (!$tournament->canRegister()) {
            $errors[] = 'Tournament registration is not open';
        }

        // Check if tournament is full
        if ($tournament->is_full) {
            $errors[] = 'Tournament is full';
        }

        return [
            'can_register' => empty($errors),
            'errors' => $errors,
        ];
    }

    /**
     * Get teams that can register for a tournament
     */
    public function getAvailableTeamsForTournament(Tournament $tournament): \Illuminate\Database\Eloquent\Collection
    {
        return Team::whereDoesntHave('tournaments', function ($query) use ($tournament) {
            $query->where('tournament_id', $tournament->id);
        })->get();
    }
} 