<?php

namespace App\Services;

use App\Models\Player;
use App\Models\TeamOfficial;
use App\Models\User;
use Illuminate\Support\Facades\Mail;
use Illuminate\Support\Facades\Log;

class OtpService
{
    /**
     * Generate and send OTP to team player
     */
    public function sendPlayerOtp(Player $player): bool
    {
        try {
            $otp = $player->generateOtp();
            
            // Send email with OTP
            Mail::send('emails.player-otp', [
                'player' => $player,
                'otp' => $otp,
                'team' => $player->team,
            ], function ($message) use ($player) {
                $message->from(config('mail.from.address'), config('mail.from.name'));
                $message->to($player->email, $player->name)
                        ->subject('Team Registration - Verify Your Email');
            });
            
            Log::info("OTP sent to player: {$player->email}");
            return true;
            
        } catch (\Exception $e) {
            Log::error("Failed to send OTP to player {$player->email}: " . $e->getMessage());
            return false;
        }
    }
    
    /**
     * Generate and send OTP to team official
     */
    public function sendOfficialOtp(TeamOfficial $official): bool
    {
        try {
            $otp = $official->generateOtp();
            
            // Send email with OTP
            Mail::send('emails.official-otp', [
                'official' => $official,
                'otp' => $otp,
                'team' => $official->team,
            ], function ($message) use ($official) {
                $message->from(config('mail.from.address'), config('mail.from.name'));
                $message->to($official->email, $official->name)
                        ->subject('Team Registration - Verify Your Email');
            });
            
            Log::info("OTP sent to official: {$official->email}");
            return true;
            
        } catch (\Exception $e) {
            Log::error("Failed to send OTP to official {$official->email}: " . $e->getMessage());
            return false;
        }
    }
    
    /**
     * Generate and send OTP to user for password change
     */
    public function sendUserOtp(User $user): bool
    {
        try {
            $otp = str_pad(random_int(0, 999999), 6, '0', STR_PAD_LEFT);
            
            $user->update([
                'otp_code' => $otp,
                'otp_expires_at' => now()->addHours(24),
            ]);
            
            // Send email with OTP
            Mail::send('emails.user-otp', [
                'user' => $user,
                'otp' => $otp,
            ], function ($message) use ($user) {
                $message->from(config('mail.from.address'), config('mail.from.name'));
                $message->to($user->email, $user->name)
                        ->subject('Password Change Required - Verify Your Email');
            });
            
            Log::info("OTP sent to user: {$user->email}");
            return true;
            
        } catch (\Exception $e) {
            Log::error("Failed to send OTP to user {$user->email}: " . $e->getMessage());
            return false;
        }
    }
    
    /**
     * Verify user OTP
     */
    public function verifyUserOtp(User $user, string $otp): bool
    {
        if ($user->otp_code === $otp && $user->otp_expires_at > now()) {
            $user->update([
                'otp_code' => null,
                'otp_expires_at' => null,
                'must_change_password' => false,
            ]);
            return true;
        }
        return false;
    }
    
    /**
     * Send OTPs to all team members
     */
    public function sendTeamOtps(\App\Models\Team $team): array
    {
        $results = [
            'players' => ['sent' => 0, 'failed' => 0],
            'officials' => ['sent' => 0, 'failed' => 0],
        ];
        
        // Send OTPs to players
        foreach ($team->teamPlayers as $player) {
            if ($this->sendPlayerOtp($player)) {
                $results['players']['sent']++;
            } else {
                $results['players']['failed']++;
            }
        }
        
        // Send OTPs to officials
        foreach ($team->teamOfficials as $official) {
            if ($this->sendOfficialOtp($official)) {
                $results['officials']['sent']++;
            } else {
                $results['officials']['failed']++;
            }
        }
        
        return $results;
    }
    
    /**
     * Check if all team members have verified their OTPs
     */
    public function isTeamFullyVerified(\App\Models\Team $team): bool
    {
        $unverifiedPlayers = $team->teamPlayers()->where('otp_verified', false)->count();
        $unverifiedOfficials = $team->teamOfficials()->where('otp_verified', false)->count();
        
        return $unverifiedPlayers === 0 && $unverifiedOfficials === 0;
    }
    
    /**
     * Get team verification status
     */
    public function getTeamVerificationStatus(\App\Models\Team $team): array
    {
        $totalPlayers = $team->teamPlayers()->count();
        $verifiedPlayers = $team->teamPlayers()->where('otp_verified', true)->count();
        
        $totalOfficials = $team->teamOfficials()->count();
        $verifiedOfficials = $team->teamOfficials()->where('otp_verified', true)->count();
        
        return [
            'players' => [
                'total' => $totalPlayers,
                'verified' => $verifiedPlayers,
                'pending' => $totalPlayers - $verifiedPlayers,
            ],
            'officials' => [
                'total' => $totalOfficials,
                'verified' => $verifiedOfficials,
                'pending' => $totalOfficials - $verifiedOfficials,
            ],
            'is_fully_verified' => $this->isTeamFullyVerified($team),
        ];
    }
} 