<?php

namespace App\Services;

use App\Models\NotificationSetting;
use Illuminate\Support\Facades\File;
use Illuminate\Support\Str;
use ReflectionClass;

class NotificationRegistryService
{
    /**
     * Scan all notification classes and sync to database
     */
    public function syncNotifications(): array
    {
        $notifications = $this->scanNotificationClasses();
        $synced = [];
        $created = 0;
        $updated = 0;

        foreach ($notifications as $notification) {
            $setting = NotificationSetting::updateOrCreate(
                ['notification_type' => $notification['notification_type']],
                [
                    'category' => $notification['category'],
                    'description' => $notification['description'],
                    'channels' => $notification['channels'],
                ]
            );

            if ($setting->wasRecentlyCreated) {
                $created++;
            } else {
                $updated++;
            }

            $synced[] = $notification['notification_type'];
        }

        // Clear cache after sync
        NotificationSetting::clearCache();

        return [
            'total' => count($notifications),
            'created' => $created,
            'updated' => $updated,
            'synced' => $synced,
        ];
    }

    /**
     * Scan all notification classes in the app/Notifications directory
     */
    public function scanNotificationClasses(): array
    {
        $notifications = [];
        $basePath = app_path('Notifications');

        // Get all PHP files recursively
        $files = File::allFiles($basePath);

        foreach ($files as $file) {
            // Skip BaseNotification
            if ($file->getFilename() === 'BaseNotification.php') {
                continue;
            }

            // Get the full class name
            $className = $this->getClassNameFromFile($file);

            if ($className && class_exists($className)) {
                $notification = $this->extractNotificationInfo($className);
                if ($notification) {
                    $notifications[] = $notification;
                }
            }
        }

        return $notifications;
    }

    /**
     * Get the full class name from a file
     */
    protected function getClassNameFromFile($file): ?string
    {
        $relativePath = str_replace(app_path(), '', $file->getPathname());
        $relativePath = str_replace(['/', '\\'], '\\', $relativePath);
        $relativePath = trim($relativePath, '\\');
        $className = 'App\\' . str_replace('.php', '', $relativePath);

        return $className;
    }

    /**
     * Extract notification information from a class
     */
    protected function extractNotificationInfo(string $className): ?array
    {
        try {
            $reflection = new ReflectionClass($className);
            
            // Skip abstract classes
            if ($reflection->isAbstract()) {
                return null;
            }

            $shortName = $reflection->getShortName();
            $namespace = $reflection->getNamespaceName();

            return [
                'notification_type' => Str::snake($shortName),
                'category' => $this->determineCategory($shortName, $namespace),
                'description' => $this->generateDescription($shortName, $namespace),
                'channels' => ['mail', 'database'],
            ];
        } catch (\Exception $e) {
            \Log::warning("Failed to extract notification info for {$className}: " . $e->getMessage());
            return null;
        }
    }

    /**
     * Determine the category of a notification based on its name and namespace
     */
    protected function determineCategory(string $shortName, string $namespace): string
    {
        // Check if it's in Admin namespace
        if (str_contains($namespace, 'Admin')) {
            return 'admin';
        }

        // Check if it's in Auth namespace
        if (str_contains($namespace, 'Auth')) {
            return 'auth';
        }

        // Determine category based on class name keywords
        $categoryKeywords = [
            'Team' => 'team',
            'Player' => 'player',
            'Payment' => 'payment',
            'Tournament' => 'tournament',
            'Match' => 'match',
            'Official' => 'official',
            'User' => 'user',
            'Certificate' => 'certificate',
            'Invitation' => 'invitation',
            'Otp' => 'auth',
            'Password' => 'auth',
            'Verification' => 'auth',
            'Email' => 'auth',
            'Bracket' => 'tournament',
            'Formation' => 'team',
            'Prize' => 'tournament',
            'Rating' => 'player',
            'Digital' => 'player',
        ];

        foreach ($categoryKeywords as $keyword => $category) {
            if (str_contains($shortName, $keyword)) {
                return $category;
            }
        }

        return 'system';
    }

    /**
     * Generate a human-readable description from the notification class name
     */
    protected function generateDescription(string $shortName, string $namespace): string
    {
        // Remove "Notification" suffix if present
        $name = str_replace('Notification', '', $shortName);
        
        // Convert camel case to words
        $words = preg_split('/(?=[A-Z])/', $name, -1, PREG_SPLIT_NO_EMPTY);
        $description = implode(' ', $words);

        // Add context based on namespace
        if (str_contains($namespace, 'Admin')) {
            $description .= ' (Admin)';
        }

        return $description;
    }

    /**
     * Get all notifications grouped by category
     */
    public function getNotificationsByCategory(): array
    {
        $notifications = $this->scanNotificationClasses();
        $grouped = [];

        foreach ($notifications as $notification) {
            $category = $notification['category'];
            if (!isset($grouped[$category])) {
                $grouped[$category] = [];
            }
            $grouped[$category][] = $notification;
        }

        // Sort categories
        ksort($grouped);

        return $grouped;
    }

    /**
     * Get notification counts by category
     */
    public function getCategoryCounts(): array
    {
        $notifications = $this->scanNotificationClasses();
        $counts = [];

        foreach ($notifications as $notification) {
            $category = $notification['category'];
            if (!isset($counts[$category])) {
                $counts[$category] = 0;
            }
            $counts[$category]++;
        }

        return $counts;
    }
}

