<?php

namespace App\Services;

use App\Models\Payment;
use App\Models\User;
use App\Models\Team;
use Illuminate\Support\Facades\Http;
use Illuminate\Support\Facades\Log;

class MpesaService
{
    protected $consumerKey;
    protected $consumerSecret;
    protected $baseUrl;
    protected $passkey;
    protected $shortcode;
    protected $callbackUrl;
    
    public function __construct()
    {
        $this->consumerKey = config('services.mpesa.consumer_key');
        $this->consumerSecret = config('services.mpesa.consumer_secret');
        $this->baseUrl = config('services.mpesa.base_url');
        $this->passkey = config('services.mpesa.passkey');
        $this->shortcode = config('services.mpesa.shortcode');
        $this->callbackUrl = config('services.mpesa.callback_url');
    }
    
    /**
     * Generate access token for M-Pesa API
     */
    public function generateAccessToken(): ?string
    {
        try {
            $credentials = base64_encode($this->consumerKey . ':' . $this->consumerSecret);
            
            $response = Http::withOptions(['verify' => false])
                ->withHeaders([
                    'Authorization' => 'Basic ' . $credentials,
                    'Content-Type' => 'application/json',
                ])
                ->get($this->baseUrl . '/oauth/v1/generate?grant_type=client_credentials');
            
            if ($response->successful()) {
                return $response->json()['access_token'];
            }
            
            Log::error('M-Pesa Access Token Error: ' . $response->body());
            return null;
        } catch (\Exception $e) {
            Log::error('M-Pesa Access Token Exception: ' . $e->getMessage());
            return null;
        }
    }
    
    /**
     * Initiate STK Push payment
     */
    public function stkPush(string $phoneNumber, float $amount, string $accountReference, string $transactionDesc = 'Payment'): array
    {
        $accessToken = $this->generateAccessToken();
        
        if (!$accessToken) {
            return ['success' => false, 'message' => 'Failed to generate access token'];
        }
        
        $timestamp = now()->format('YmdHis');
        $password = base64_encode($this->shortcode . $this->passkey . $timestamp);
        
        $payload = [
            'BusinessShortCode' => $this->shortcode,
            'Password' => $password,
            'Timestamp' => $timestamp,
            'TransactionType' => 'CustomerPayBillOnline',
            'Amount' => $amount,
            'PartyA' => $this->formatPhoneNumber($phoneNumber),
            'PartyB' => $this->shortcode,
            'PhoneNumber' => $this->formatPhoneNumber($phoneNumber),
            'CallBackURL' => $this->callbackUrl,
            'AccountReference' => $accountReference,
            'TransactionDesc' => $transactionDesc,
        ];
        
        try {
            $response = Http::withOptions(['verify' => false])
                ->withHeaders([
                    'Authorization' => 'Bearer ' . $accessToken,
                    'Content-Type' => 'application/json',
                ])
                ->post($this->baseUrl . '/mpesa/stkpush/v1/processrequest', $payload);
                
            $responseData = $response->json();

            Log::info('M-Pesa STK Push Response: ' . $response->body());
            
            if ($response->successful() && $responseData['ResponseCode'] == '0') {
                return [
                    'success' => true,
                    'checkout_request_id' => $responseData['CheckoutRequestID'],
                    'merchant_request_id' => $responseData['MerchantRequestID'],
                    'customer_message' => $responseData['CustomerMessage'],
                ];
            }
            
            Log::error('M-Pesa STK Push Error: ' . $response->body());
            return [
                'success' => false,
                'message' => $responseData['errorMessage'] ?? 'Payment initiation failed'
            ];
            
        } catch (\Exception $e) {
            Log::error('M-Pesa STK Push Exception: ' . $e->getMessage());
            return ['success' => false, 'message' => 'Payment service unavailable'];
        }
    }
    
    /**
     * Query STK Push status
     */
    public function queryStkPush(string $checkoutRequestId): array
    {
        $accessToken = $this->generateAccessToken();
        
        if (!$accessToken) {
            return ['success' => false, 'message' => 'Failed to generate access token'];
        }
        
        $timestamp = now()->format('YmdHis');
        $password = base64_encode($this->shortcode . $this->passkey . $timestamp);
        
        $payload = [
            'BusinessShortCode' => $this->shortcode,
            'Password' => $password,
            'Timestamp' => $timestamp,
            'CheckoutRequestID' => $checkoutRequestId,
        ];
        
        try {
            $response = Http::withOptions(['verify' => false])
                ->withHeaders([
                    'Authorization' => 'Bearer ' . $accessToken,
                    'Content-Type' => 'application/json',
                ])
                ->post($this->baseUrl . '/mpesa/stkpushquery/v1/query', $payload);

                Log::info('M-Pesa Query Response: ' . $response->body());
                
            return $response->json();
            
        } catch (\Exception $e) {
            Log::error('M-Pesa Query Exception: ' . $e->getMessage());
            return ['success' => false, 'message' => 'Query service unavailable'];
        }
    }
    
    /**
     * Process M-Pesa callback
     */
    public function processCallback(array $callbackData): bool
    {
        try {
            $checkoutRequestId = $callbackData['Body']['stkCallback']['CheckoutRequestID'] ?? null;
            $resultCode = $callbackData['Body']['stkCallback']['ResultCode'] ?? null;
            
            if (!$checkoutRequestId) {
                Log::error('M-Pesa Callback: Missing CheckoutRequestID');
                return false;
            }
            
            $payment = Payment::where('checkout_request_id', $checkoutRequestId)->first();
            
            if (!$payment) {
                Log::error('M-Pesa Callback: Payment not found for CheckoutRequestID: ' . $checkoutRequestId);
                return false;
            }
            
            if ($resultCode == 0) {
                // Payment successful
                $callbackMetadata = $callbackData['Body']['stkCallback']['CallbackMetadata']['Item'] ?? [];
                $mpesaReceiptNumber = collect($callbackMetadata)->firstWhere('Name', 'MpesaReceiptNumber')['Value'] ?? null;
                
                $payment->update([
                    'status' => 'completed',
                    'paid_at' => now(),
                    'reference_number' => $mpesaReceiptNumber,
                    'payment_data' => $callbackData,
                ]);
                
                // Update user payment status
                $payment->user->update(['payment_status' => 'completed']);
                
                Log::info('M-Pesa Payment Successful: ' . $checkoutRequestId);
                return true;
                
            } else {
                // Payment failed
                $errorMessage = $callbackData['Body']['stkCallback']['ResultDesc'] ?? 'Payment failed';
                
                $payment->update([
                    'status' => 'failed',
                    'failure_reason' => $errorMessage,
                    'payment_data' => $callbackData,
                ]);
                
                Log::info('M-Pesa Payment Failed: ' . $checkoutRequestId . ' - ' . $errorMessage);
                return false;
            }
            
        } catch (\Exception $e) {
            Log::error('M-Pesa Callback Exception: ' . $e->getMessage());
            return false;
        }
    }
    
    /**
     * Create payment record and initiate payment
     */
    public function createPayment(User $user, float $amount, ?Team $team = null, string $description = 'Tournament Registration', ?string $phoneNumber = null): array
    {
        try {
            \Log::info('MpesaService: Creating payment', [
                'user_id' => $user->id,
                'team_id' => $team?->id,
                'team_category' => $team?->category,
                'amount' => $amount,
                'description' => $description,
            ]);
            
            // Create payment record
            $payment = Payment::create([
                'user_id' => $user->id,
                'team_id' => $team?->id,
                'payment_method' => 'mpesa',
                'amount' => $amount,
                'currency' => 'KES',
                'reference_number' => uniqid('PAY_'),
                'description' => $description,
                'status' => 'pending',
            ]);
            
            // Use provided phone number or fall back to user's phone
            $phone = $phoneNumber ?: $user->phone;
            
            if (!$phone) {
                throw new \Exception('Phone number is required for M-Pesa payment');
            }
            
            \Log::info('MpesaService: Initiating STK Push', [
                'payment_id' => $payment->id,
                'phone' => substr($phone, 0, 6) . 'XXX', // Masked for security
                'amount' => $amount,
                'reference' => $payment->reference_number,
            ]);
            
            // Initiate STK Push
            $stkResult = $this->stkPush(
                $phone,
                $amount,
                $payment->reference_number,
                $description
            );
            
            if ($stkResult['success']) {
                $payment->update([
                    'checkout_request_id' => $stkResult['checkout_request_id'],
                    'merchant_request_id' => $stkResult['merchant_request_id'],
                    'status' => 'processing',
                ]);
                
                return [
                    'success' => true,
                    'payment' => $payment,
                    'message' => $stkResult['customer_message'],
                ];
            }
            
            $payment->markAsFailed($stkResult['message']);
            return $stkResult;
            
        } catch (\Exception $e) {
            Log::error('Create Payment Exception: ' . $e->getMessage());
            return ['success' => false, 'message' => 'Payment creation failed'];
        }
    }
    
    /**
     * Format phone number for M-Pesa
     */
    protected function formatPhoneNumber(string $phoneNumber): string
    {
        // Remove any non-numeric characters
        $phoneNumber = preg_replace('/[^0-9]/', '', $phoneNumber);
        
        // Add country code if missing
        if (substr($phoneNumber, 0, 3) !== '254') {
            if (substr($phoneNumber, 0, 1) === '0') {
                $phoneNumber = '254' . substr($phoneNumber, 1);
            } else {
                $phoneNumber = '254' . $phoneNumber;
            }
        }
        
        return $phoneNumber;
    }
} 