<?php

namespace App\Services;

use Illuminate\Support\Facades\Mail;
use Illuminate\Support\Facades\Log;
use Illuminate\Support\Facades\Config;
use Illuminate\Support\Facades\Validator;
use Exception;

class EmailSettingsService
{
    /**
     * Test email configuration by sending a test email
     */
    public function sendTestEmail(string $toEmail, string $toName = null, array $settings = []): array
    {
        try {
            // Temporarily update mail configuration if settings provided
            if (!empty($settings)) {
                $this->applyTemporaryMailSettings($settings);
            }

            $testSubject = config('mail.test.subject_prefix', '[TEST]') . ' Email Configuration Test';
            $testData = [
                'app_name' => config('app.name'),
                'test_time' => now()->format('Y-m-d H:i:s'),
                'to_name' => $toName ?: 'Test User',
                'settings_used' => $this->getCurrentMailSettings(),
            ];

            Mail::send('emails.test-email', $testData, function ($message) use ($toEmail, $toName, $testSubject) {
                $message->to($toEmail, $toName ?: 'Test User')
                        ->subject($testSubject);
            });

            Log::info("Test email sent successfully to: {$toEmail}");

            return [
                'success' => true,
                'message' => 'Test email sent successfully!',
                'details' => [
                    'to' => $toEmail,
                    'subject' => $testSubject,
                    'mailer' => config('mail.default'),
                    'sent_at' => now()->format('Y-m-d H:i:s'),
                ]
            ];

        } catch (Exception $e) {
            Log::error("Failed to send test email to {$toEmail}: " . $e->getMessage());

            return [
                'success' => false,
                'message' => 'Failed to send test email: ' . $e->getMessage(),
                'error_details' => [
                    'exception' => get_class($e),
                    'line' => $e->getLine(),
                    'file' => basename($e->getFile()),
                ]
            ];
        }
    }

    /**
     * Validate email settings before saving
     */
    public function validateEmailSettings(array $settings): array
    {
        $validator = Validator::make($settings, [
            'mail_driver' => 'required|string|in:smtp,mailgun,ses,sendmail,log,postmark,resend',
            'mail_host' => 'required_if:mail_driver,smtp|string|max:255',
            'mail_port' => 'required_if:mail_driver,smtp|integer|between:1,65535',
            'mail_username' => 'nullable|string|max:255',
            'mail_password' => 'nullable|string|max:255',
            'mail_encryption' => 'nullable|string|in:tls,ssl,starttls',
            'mail_from_address' => 'required|email|max:255',
            'mail_from_name' => 'required|string|max:255',
            'mail_timeout' => 'nullable|integer|between:10,300',
        ]);

        if ($validator->fails()) {
            return [
                'valid' => false,
                'errors' => $validator->errors()->toArray(),
                'message' => 'Email settings validation failed.'
            ];
        }

        // Additional custom validations
        $customValidations = $this->performCustomValidations($settings);
        if (!$customValidations['valid']) {
            return $customValidations;
        }

        return [
            'valid' => true,
            'message' => 'Email settings are valid.',
            'normalized_settings' => $this->normalizeEmailSettings($settings)
        ];
    }

    /**
     * Test SMTP connection without sending email
     */
    public function testSmtpConnection(array $settings): array
    {
        if ($settings['mail_driver'] !== 'smtp') {
            return [
                'success' => false,
                'message' => 'SMTP connection test is only available for SMTP driver.'
            ];
        }

        try {
            // Test basic socket connection to SMTP server
            $timeout = $settings['mail_timeout'] ?? 60;
            $host = $settings['mail_host'];
            $port = $settings['mail_port'];
            
            // Create socket connection
            $socket = @fsockopen($host, $port, $errno, $errstr, $timeout);
            
            if (!$socket) {
                return [
                    'success' => false,
                    'message' => "SMTP connection failed: {$errstr} ({$errno})",
                    'error_details' => [
                        'errno' => $errno,
                        'errstr' => $errstr,
                    ]
                ];
            }

            // Read server greeting
            $response = fgets($socket, 512);
            
            if (!$response || !str_starts_with($response, '220')) {
                fclose($socket);
                return [
                    'success' => false,
                    'message' => 'SMTP server did not respond with proper greeting',
                    'error_details' => [
                        'response' => trim($response ?: 'No response'),
                    ]
                ];
            }

            // Send EHLO command
            fwrite($socket, "EHLO " . parse_url(config('app.url'), PHP_URL_HOST) . "\r\n");
            $response = fgets($socket, 512);

            // Close connection
            fwrite($socket, "QUIT\r\n");
            fclose($socket);

            return [
                'success' => true,
                'message' => 'SMTP connection successful!',
                'details' => [
                    'host' => $host,
                    'port' => $port,
                    'encryption' => $settings['mail_encryption'] ?? 'none',
                    'authenticated' => !empty($settings['mail_username']),
                    'greeting' => trim($response),
                ]
            ];

        } catch (Exception $e) {
            return [
                'success' => false,
                'message' => 'SMTP connection failed: ' . $e->getMessage(),
                'error_details' => [
                    'exception' => get_class($e),
                    'code' => $e->getCode(),
                ]
            ];
        }
    }

    /**
     * Get current mail configuration settings
     */
    public function getCurrentMailSettings(): array
    {
        return [
            'mail_driver' => config('mail.default'),
            'mail_host' => config('mail.mailers.smtp.host'),
            'mail_port' => config('mail.mailers.smtp.port'),
            'mail_encryption' => config('mail.mailers.smtp.encryption'),
            'mail_username' => config('mail.mailers.smtp.username'),
            'mail_from_address' => config('mail.from.address'),
            'mail_from_name' => config('mail.from.name'),
            'mail_timeout' => config('mail.mailers.smtp.timeout'),
        ];
    }

    /**
     * Apply temporary mail settings for testing
     */
    private function applyTemporaryMailSettings(array $settings): void
    {
        if (isset($settings['mail_driver'])) {
            Config::set('mail.default', $settings['mail_driver']);
        }

        if (isset($settings['mail_host'])) {
            Config::set('mail.mailers.smtp.host', $settings['mail_host']);
        }

        if (isset($settings['mail_port'])) {
            Config::set('mail.mailers.smtp.port', $settings['mail_port']);
        }

        if (isset($settings['mail_encryption'])) {
            Config::set('mail.mailers.smtp.encryption', $settings['mail_encryption']);
        }

        if (isset($settings['mail_username'])) {
            Config::set('mail.mailers.smtp.username', $settings['mail_username']);
        }

        if (isset($settings['mail_password'])) {
            Config::set('mail.mailers.smtp.password', $settings['mail_password']);
        }

        if (isset($settings['mail_from_address'])) {
            Config::set('mail.from.address', $settings['mail_from_address']);
        }

        if (isset($settings['mail_from_name'])) {
            Config::set('mail.from.name', $settings['mail_from_name']);
        }

        if (isset($settings['mail_timeout'])) {
            Config::set('mail.mailers.smtp.timeout', $settings['mail_timeout']);
        }
    }

    /**
     * Perform custom validations for email settings
     */
    private function performCustomValidations(array $settings): array
    {
        $errors = [];

        // Validate SMTP settings if SMTP is selected
        if ($settings['mail_driver'] === 'smtp') {
            if (empty($settings['mail_host'])) {
                $errors['mail_host'] = ['SMTP host is required when using SMTP driver.'];
            }

            if (empty($settings['mail_port'])) {
                $errors['mail_port'] = ['SMTP port is required when using SMTP driver.'];
            }

            // Validate common SMTP ports
            if (!empty($settings['mail_port'])) {
                $commonPorts = [25, 465, 587, 2525];
                if (!in_array((int)$settings['mail_port'], $commonPorts)) {
                    $errors['mail_port'] = ['Consider using standard SMTP ports: 25, 465, 587, or 2525.'];
                }
            }

            // Validate encryption settings
            if (!empty($settings['mail_encryption'])) {
                if ($settings['mail_port'] == 465 && $settings['mail_encryption'] !== 'ssl') {
                    $errors['mail_encryption'] = ['Port 465 typically requires SSL encryption.'];
                } elseif ($settings['mail_port'] == 587 && !in_array($settings['mail_encryption'], ['tls', 'starttls'])) {
                    $errors['mail_encryption'] = ['Port 587 typically requires TLS or STARTTLS encryption.'];
                }
            }
        }

        if (!empty($errors)) {
            return [
                'valid' => false,
                'errors' => $errors,
                'message' => 'Email settings have validation issues.'
            ];
        }

        return ['valid' => true];
    }

    /**
     * Normalize email settings for consistency
     */
    private function normalizeEmailSettings(array $settings): array
    {
        $normalized = $settings;

        // Normalize boolean values
        if (isset($normalized['mail_verify_peer'])) {
            $normalized['mail_verify_peer'] = filter_var($normalized['mail_verify_peer'], FILTER_VALIDATE_BOOLEAN);
        }

        // Normalize numeric values
        if (isset($normalized['mail_port'])) {
            $normalized['mail_port'] = (int) $normalized['mail_port'];
        }

        if (isset($normalized['mail_timeout'])) {
            $normalized['mail_timeout'] = (int) $normalized['mail_timeout'];
        }

        // Trim string values
        $stringFields = ['mail_host', 'mail_username', 'mail_password', 'mail_from_address', 'mail_from_name'];
        foreach ($stringFields as $field) {
            if (isset($normalized[$field])) {
                $normalized[$field] = trim($normalized[$field]);
            }
        }

        return $normalized;
    }
}