<?php

namespace App\Services;

use App\Models\User;
use App\Models\Tournament;
use App\Models\Player;
use App\Models\Team;
use App\Models\GameMatch;
use Illuminate\Database\Eloquent\Builder;

class DataScopingService
{
    /**
     * Scope tournaments based on user role
     */
    public function scopeTournaments(Builder $query, User $user): Builder
    {
        $userRoles = $user->roles()->pluck('name')->toArray();

        // Admins can see all tournaments
        if (in_array('admin', $userRoles) || in_array('super_admin', $userRoles)) {
            return $query;
        }

        // Organizers can see tournaments they manage
        if (in_array('organizer', $userRoles)) {
            return $query->where('organizer_id', $user->id);
        }

        // Referees can see tournaments they're assigned to
        if (in_array('referee', $userRoles)) {
            return $query->whereHas('matches', function($q) use ($user) {
                $q->where('referee_id', $user->id);
            });
        }

        // Players can see tournaments they're participating in
        if (in_array('player', $userRoles)) {
            $player = Player::where('user_id', $user->id)->first();
            if ($player) {
                return $query->whereHas('teams.players', function($q) use ($player) {
                    $q->where('player_id', $player->id);
                });
            }
        }

        return $query->where('id', 0);
    }

    /**
     * Scope matches based on user role
     */
    public function scopeMatches(Builder $query, User $user): Builder
    {
        $userRoles = $user->roles()->pluck('name')->toArray();

        // Admins can see all matches
        if (in_array('admin', $userRoles) || in_array('super_admin', $userRoles)) {
            return $query;
        }

        // Organizers can see matches in their tournaments
        if (in_array('organizer', $userRoles)) {
            $tournamentIds = Tournament::where('organizer_id', $user->id)->pluck('id');
            return $query->whereIn('tournament_id', $tournamentIds);
        }

        // Referees can see matches they're officiating
        if (in_array('referee', $userRoles)) {
            return $query->where('referee_id', $user->id);
        }

        // Players can see matches they're involved in
        if (in_array('player', $userRoles)) {
            $player = Player::where('user_id', $user->id)->first();
            if ($player) {
                return $query->where(function($q) use ($player) {
                    $q->whereHas('team1.players', function($subQ) use ($player) {
                        $subQ->where('player_id', $player->id);
                    })->orWhereHas('team2.players', function($subQ) use ($player) {
                        $subQ->where('player_id', $player->id);
                    });
                });
            }
        }

        return $query->where('id', 0);
    }

    /**
     * Check if user can access specific resource
     */
    public function canAccessResource(User $user, string $resource, int $resourceId): bool
    {
        $userRoles = $user->roles()->pluck('name')->toArray();

        // Admins can access everything
        if (in_array('admin', $userRoles) || in_array('super_admin', $userRoles)) {
            return true;
        }

        switch ($resource) {
            case 'tournament':
                return $this->canAccessTournament($user, $resourceId);
            case 'match':
                return $this->canAccessMatch($user, $resourceId);
            default:
                return false;
        }
    }

    /**
     * Check if user can access specific tournament
     */
    private function canAccessTournament(User $user, int $tournamentId): bool
    {
        $userRoles = $user->roles()->pluck('name')->toArray();
        $tournament = Tournament::find($tournamentId);

        if (!$tournament) {
            return false;
        }

        // Organizers can access tournaments they manage
        if (in_array('organizer', $userRoles)) {
            return $tournament->organizer_id === $user->id;
        }

        // Referees can access tournaments they're assigned to
        if (in_array('referee', $userRoles)) {
            return $tournament->matches()->where('referee_id', $user->id)->exists();
        }

        return false;
    }

    /**
     * Check if user can access specific match
     */
    private function canAccessMatch(User $user, int $matchId): bool
    {
        $userRoles = $user->roles()->pluck('name')->toArray();
        $match = GameMatch::find($matchId);

        if (!$match) {
            return false;
        }

        // Referees can access matches they're officiating
        if (in_array('referee', $userRoles)) {
            return $match->referee_id === $user->id;
        }

        // Organizers can access matches in their tournaments
        if (in_array('organizer', $userRoles)) {
            return $match->tournament->organizer_id === $user->id;
        }

        return false;
    }
} 