<?php

namespace App\Services;

use Carbon\Carbon;

class CategoryValidationService
{
    /**
     * Validate if a player's age meets category requirements
     */
    public function validatePlayerAge(?string $dob, string $category, ?Carbon $tournamentStartDate = null): array
    {
        if (!$dob) {
            return [
                'valid' => false,
                'error' => 'Date of birth is required for category validation'
            ];
        }

        $age = $this->calculatePlayerAge($dob, $tournamentStartDate);
        $requirements = $this->getCategoryRequirements($category);

        $ageMin = $requirements['age_min'];
        $ageMax = $requirements['age_max'];

        // Check minimum age
        if ($ageMin !== null && $age < $ageMin) {
            return [
                'valid' => false,
                'error' => "Player is too young for {$requirements['name']} category (Age: {$age}, Minimum: {$ageMin} years)"
            ];
        }

        // Check maximum age
        if ($ageMax !== null && $age > $ageMax) {
            return [
                'valid' => false,
                'error' => "Player is too old for {$requirements['name']} category (Age: {$age}, Maximum: {$ageMax} years)"
            ];
        }

        return ['valid' => true];
    }

    /**
     * Validate if a player's gender meets category requirements
     */
    public function validatePlayerGender(?string $gender, string $category): array
    {
        $requirements = $this->getCategoryRequirements($category);

        // If gender not required for category (e.g., children), skip
        if (!$requirements['gender_required']) {
            return ['valid' => true];
        }

        if (!$gender) {
            return [
                'valid' => false,
                'error' => 'Gender is required for category validation'
            ];
        }

        $requiredGender = $requirements['gender'];
        
        if ($requiredGender && strtolower($gender) !== strtolower($requiredGender)) {
            return [
                'valid' => false,
                'error' => "Player gender does not match {$requirements['name']} category requirements (Required: {$requiredGender}, Provided: {$gender})"
            ];
        }

        return ['valid' => true];
    }

    /**
     * Validate if a team's players all match the category
     */
    public function validateTeamCategory($team, ?Carbon $tournamentStartDate = null): array
    {
        $category = $team->category ?? 'men';
        $errors = [];
        $players = $team->players;

        foreach ($players as $player) {
            // Validate age
            $ageValidation = $this->validatePlayerAge($player->date_of_birth, $category, $tournamentStartDate);
            if (!$ageValidation['valid']) {
                $errors[] = "{$player->display_name}: {$ageValidation['error']}";
            }

            // Validate gender
            $genderValidation = $this->validatePlayerGender($player->gender, $category);
            if (!$genderValidation['valid']) {
                $errors[] = "{$player->display_name}: {$genderValidation['error']}";
            }
        }

        return [
            'valid' => empty($errors),
            'errors' => $errors
        ];
    }

    /**
     * Get category requirements
     */
    public function getCategoryRequirements(string $category): array
    {
        $categories = config('tournament.categories', []);
        
        return $categories[$category] ?? [
            'name' => ucfirst(str_replace('_', ' ', $category)),
            'age_min' => null,
            'age_max' => null,
            'gender' => null,
            'gender_required' => false,
            'default_fee' => config('tournament.registration_fee', 1000),
            'icon' => '👥',
            'color' => 'gray',
            'description' => 'General category',
        ];
    }

    /**
     * Calculate player's age
     */
    public function calculatePlayerAge(string $dob, ?Carbon $referenceDate = null): int
    {
        $birthDate = Carbon::parse($dob);
        $reference = $referenceDate ?? Carbon::now();
        
        return $birthDate->diffInYears($reference);
    }

    /**
     * Get all available categories
     */
    public function getAllCategories(): array
    {
        return config('tournament.categories', []);
    }

    /**
     * Get suggested category based on players' ages and genders
     */
    public function getSuggestedCategory(array $players, ?Carbon $tournamentStartDate = null): ?string
    {
        if (empty($players)) {
            return null;
        }

        // Calculate ages
        $ages = [];
        $genders = [];

        foreach ($players as $player) {
            if (!empty($player['date_of_birth'])) {
                $ages[] = $this->calculatePlayerAge($player['date_of_birth'], $tournamentStartDate);
            }
            if (!empty($player['gender'])) {
                $genders[] = strtolower($player['gender']);
            }
        }

        if (empty($ages) || empty($genders)) {
            return null;
        }

        $minAge = min($ages);
        $maxAge = max($ages);
        $allMale = count(array_unique($genders)) === 1 && $genders[0] === 'male';
        $allFemale = count(array_unique($genders)) === 1 && $genders[0] === 'female';

        // Suggest based on age ranges and gender
        if ($maxAge <= 12) {
            return 'children';
        } elseif ($minAge >= 13 && $maxAge <= 17) {
            return $allMale ? 'youth_male' : ($allFemale ? 'youth_female' : null);
        } elseif ($minAge >= 18) {
            return $allMale ? 'men' : ($allFemale ? 'women' : null);
        }

        return null;
    }

    /**
     * Validate a single player data array against category
     */
    public function validatePlayerData(array $playerData, string $category, ?Carbon $tournamentStartDate = null): array
    {
        $errors = [];

        // Validate age
        if (isset($playerData['date_of_birth'])) {
            $ageValidation = $this->validatePlayerAge($playerData['date_of_birth'], $category, $tournamentStartDate);
            if (!$ageValidation['valid']) {
                $errors[] = $ageValidation['error'];
            }
        } else {
            $errors[] = 'Date of birth is required';
        }

        // Validate gender
        if (isset($playerData['gender'])) {
            $genderValidation = $this->validatePlayerGender($playerData['gender'], $category);
            if (!$genderValidation['valid']) {
                $errors[] = $genderValidation['error'];
            }
        } else {
            $requirements = $this->getCategoryRequirements($category);
            if ($requirements['gender_required']) {
                $errors[] = 'Gender is required';
            }
        }

        return [
            'valid' => empty($errors),
            'errors' => $errors
        ];
    }

    /**
     * Validate category change impact on existing teams
     */
    public function validateCategoryChange(\App\Models\Tournament $tournament, array $newCategorySettings): array
    {
        $issues = [];
        $teams = $tournament->teams;
        
        foreach ($teams as $team) {
            // Check if team's category still available
            if (!in_array($team->category, $newCategorySettings['available_categories'] ?? [])) {
                $issues[] = "Team '{$team->name}' is in category '{$team->category}' which would be disabled";
            }
            
            // Check fee changes
            if (isset($newCategorySettings['category_fees'][$team->category])) {
                $oldFee = $tournament->getFeeForCategory($team->category);
                $newFee = $newCategorySettings['category_fees'][$team->category];
                if ($oldFee != $newFee) {
                    $issues[] = "Team '{$team->name}' fee would change from {$oldFee} to {$newFee}";
                }
            }
        }
        
        return [
            'valid' => empty($issues),
            'issues' => $issues,
            'teams_affected' => $teams->count(),
        ];
    }
}

