<?php

namespace App\Observers;

use App\Models\User;
use App\Models\Role;
use App\Notifications\User\UserProfileUpdatedNotification;
use App\Notifications\User\UserRoleChangedNotification;
use App\Notifications\User\UserPasswordChangedNotification;
use App\Notifications\User\UserPaymentStatusUpdatedNotification;
use App\Notifications\User\UserRegisteredNotification;
use App\Notifications\Admin\UserRegisteredAdminNotification;
use App\Notifications\Admin\UserRoleChangedAdminNotification;
use App\Notifications\Admin\UserPasswordChangedAdminNotification;
use App\Notifications\Admin\UserProfileUpdatedAdminNotification;
use App\Notifications\Admin\UserPaymentStatusUpdatedAdminNotification;
use Illuminate\Support\Facades\Notification;

class UserObserver
{
    // Static properties to temporarily store registration data
    private static $tempPassword = null;
    private static $tempUsername = null;

    /**
     * Set temporary registration data before user creation
     */
    public static function setTempRegistrationData(string $password = null, string $username = null): void
    {
        self::$tempPassword = $password;
        self::$tempUsername = $username;
    }

    /**
     * Clear temporary registration data after use
     */
    public static function clearTempRegistrationData(): void
    {
        self::$tempPassword = null;
        self::$tempUsername = null;
    }

    /**
     * Handle the User "created" event.
     */
    public function created(User $user): void
    {
        // Get temporary data and clear it
        $password = self::$tempPassword;
        $username = self::$tempUsername;
        self::clearTempRegistrationData();

        // Assign default role based on user_type
        $this->assignDefaultRole($user);

        // Notify the user about successful registration with password and username
        $user->notify(new UserRegisteredNotification($user, $password, $username));

        // Notify admins about new user registration
        $this->notifyAdmins(new UserRegisteredAdminNotification($user));

        // Log the event
        \Log::info('User created', [
            'user_id' => $user->id,
            'email' => $user->email,
            'user_type' => $user->user_type,
            'created_at' => $user->created_at,
        ]);
    }

    /**
     * Handle the User "updated" event.
     */
    public function updated(User $user): void
    {
        $changes = $user->getChanges();
        $original = $user->getOriginal();

        // Handle profile updates
        if ($this->hasProfileChanges($changes)) {
            $user->notify(new UserProfileUpdatedNotification($user, $changes));
            // $this->notifyAdmins(new UserProfileUpdatedAdminNotification($user, $changes));
        }

        // Handle role changes
        if (isset($changes['user_type']) && $changes['user_type'] !== $original['user_type']) {
            // Update user roles based on new user_type
            $this->updateUserRoles($user, $original['user_type'], $changes['user_type']);
            
            $user->notify(new UserRoleChangedNotification($user, $original['user_type'], $changes['user_type']));
            $this->notifyAdmins(new UserRoleChangedAdminNotification($user, $original['user_type'], $changes['user_type']));
        }

        // Handle password changes
        if (isset($changes['password'])) {
            $user->notify(new UserPasswordChangedNotification($user));
            // $this->notifyAdmins(new UserPasswordChangedAdminNotification($user));
        }

        // Handle payment status changes
        if (isset($changes['payment_status']) && $changes['payment_status'] !== $original['payment_status']) {
            $user->notify(new UserPaymentStatusUpdatedNotification($user, $original['payment_status'], $changes['payment_status']));
            $this->notifyAdmins(new UserPaymentStatusUpdatedAdminNotification($user, $original['payment_status'], $changes['payment_status']));
        }

        // Log the event
        \Log::info('User updated', [
            'user_id' => $user->id,
            'email' => $user->email,
            'changes' => array_keys($changes),
            'updated_at' => $user->updated_at,
        ]);
    }

    /**
     * Handle the User "deleted" event.
     */
    public function deleted(User $user): void
    {
        // Notify admins about user deletion
        $this->notifyAdmins(new \App\Notifications\Admin\UserDeletedAdminNotification($user));

        // Log the event
        \Log::info('User deleted', [
            'user_id' => $user->id,
            'email' => $user->email,
            'deleted_at' => now(),
        ]);
    }

    /**
     * Handle the User "restored" event.
     */
    public function restored(User $user): void
    {
        // Notify the user about account restoration
        $user->notify(new \App\Notifications\User\UserRestoredNotification($user));

        // Notify admins about user restoration
        $this->notifyAdmins(new \App\Notifications\Admin\UserRestoredAdminNotification($user));

        // Log the event
        \Log::info('User restored', [
            'user_id' => $user->id,
            'email' => $user->email,
            'restored_at' => now(),
        ]);
    }

    /**
     * Handle the User "force deleted" event.
     */
    public function forceDeleted(User $user): void
    {
        // Notify admins about user force deletion
        $this->notifyAdmins(new \App\Notifications\Admin\UserForceDeletedAdminNotification($user));

        // Log the event
        \Log::info('User force deleted', [
            'user_id' => $user->id,
            'email' => $user->email,
            'force_deleted_at' => now(),
        ]);
    }

    /**
     * Assign default role based on user_type
     */
    private function assignDefaultRole(User $user): void
    {
        try {
            $roleName = $this->getDefaultRoleName($user->user_type);
            
            if ($roleName) {
                $role = Role::where('name', $roleName)->first();
                
                if ($role) {
                    $user->assignRole($role, true); // true = is_primary
                    
                    \Log::info('Default role assigned to user', [
                        'user_id' => $user->id,
                        'user_type' => $user->user_type,
                        'role_name' => $roleName,
                    ]);
                } else {
                    \Log::warning('Default role not found for user_type', [
                        'user_id' => $user->id,
                        'user_type' => $user->user_type,
                        'expected_role' => $roleName,
                    ]);
                }
            }
        } catch (\Exception $e) {
            \Log::error('Failed to assign default role to user', [
                'user_id' => $user->id,
                'user_type' => $user->user_type,
                'error' => $e->getMessage(),
            ]);
        }
    }

    /**
     * Get default role name based on user_type
     */
    private function getDefaultRoleName(?string $userType): ?string
    {
        return match ($userType) {
            'player' => 'player',
            'team' => 'player', // Team captains are players
            'individual' => 'player', // Individual players are players
            'spectator' => 'spectator',
            'referee' => 'referee',
            'organizer' => 'organizer',
            'admin' => 'admin',
            'super_admin' => 'super_admin',
            'official' => 'referee', // Team officials are referees
            default => 'spectator', // Default fallback role
        };
    }

    /**
     * Update user roles when user_type changes
     */
    private function updateUserRoles(User $user, string $oldUserType, string $newUserType): void
    {
        try {
            $newRoleName = $this->getDefaultRoleName($newUserType);
            
            if ($newRoleName) {
                $newRole = Role::where('name', $newRoleName)->first();
                
                if ($newRole) {
                    // Remove old primary role if it exists
                    $oldPrimaryRole = $user->getPrimaryRole();
                    if ($oldPrimaryRole) {
                        $user->removeRole($oldPrimaryRole);
                    }
                    
                    // Assign new primary role
                    $user->assignRole($newRole, true); // true = is_primary
                    
                    \Log::info('User roles updated due to user_type change', [
                        'user_id' => $user->id,
                        'old_user_type' => $oldUserType,
                        'new_user_type' => $newUserType,
                        'new_role_name' => $newRoleName,
                    ]);
                } else {
                    \Log::warning('Role not found for user_type change', [
                        'user_id' => $user->id,
                        'new_user_type' => $newUserType,
                        'expected_role' => $newRoleName,
                    ]);
                }
            }
        } catch (\Exception $e) {
            \Log::error('Failed to update user roles after user_type change', [
                'user_id' => $user->id,
                'old_user_type' => $oldUserType,
                'new_user_type' => $newUserType,
                'error' => $e->getMessage(),
            ]);
        }
    }

    /**
     * Check if the changes include profile-related fields.
     */
    private function hasProfileChanges(array $changes): bool
    {
        $profileFields = ['name', 'phone', 'avatar', 'bio', 'date_of_birth', 'gender', 'nationality'];
        return !empty(array_intersect(array_keys($changes), $profileFields));
    }

    /**
     * Notify all admin users.
     */
    private function notifyAdmins($notification): void
    {
        // Notify both admin and super_admin users
        $admins = User::whereHas('roles', function ($query) {
            $query->whereIn('name', ['admin', 'super_admin', 'organizer']);
        })->get();

        Notification::send($admins, $notification);
    }
}