<?php

namespace App\Observers;

use App\Models\Tournament;
use App\Models\User;
use App\Notifications\User\TournamentCreatedNotification;
use App\Notifications\User\TournamentUpdatedNotification;
use App\Notifications\User\TournamentStatusChangedNotification;
use App\Notifications\User\TournamentRegistrationOpenedNotification;
use App\Notifications\User\TournamentRegistrationClosedNotification;
use App\Notifications\User\TournamentStartedNotification;
use App\Notifications\User\TournamentCompletedNotification;
use App\Notifications\User\TournamentCancelledNotification;
use App\Notifications\User\TournamentPublishedNotification;
use App\Notifications\Admin\TournamentCreatedAdminNotification;
use App\Notifications\Admin\TournamentUpdatedAdminNotification;
use App\Notifications\Admin\TournamentStatusChangedAdminNotification;
use App\Notifications\Admin\TournamentRegistrationOpenedAdminNotification;
use App\Notifications\Admin\TournamentRegistrationClosedAdminNotification;
use App\Notifications\Admin\TournamentStartedAdminNotification;
use App\Notifications\Admin\TournamentCompletedAdminNotification;
use App\Notifications\Admin\TournamentCancelledAdminNotification;
use App\Notifications\Admin\TournamentPublishedAdminNotification;
use Illuminate\Support\Facades\Notification;

class TournamentObserver
{
    /**
     * Handle the Tournament "created" event.
     */
    public function created(Tournament $tournament): void
    {
        // Notify all admins and organizers about new tournament creation
        $this->notifyAdmins(new TournamentCreatedAdminNotification($tournament));

        // If tournament is immediately published, notify users
        if ($tournament->published_at) {
            $this->notifyAllUsers(new TournamentCreatedNotification($tournament));
        }

        // Log the event
        \Log::info('Tournament created', [
            'tournament_id' => $tournament->id,
            'name' => $tournament->name,
            'slug' => $tournament->slug,
            'type' => $tournament->type,
            'status' => $tournament->status,
            'start_date' => $tournament->start_date,
            'end_date' => $tournament->end_date,
            'created_at' => $tournament->created_at,
        ]);
    }

    /**
     * Handle the Tournament "updated" event.
     */
    // public function updated(Tournament $tournament): void
    // {
    //     $changes = $tournament->getChanges();
    //     $original = $tournament->getOriginal();

    //     // Handle status changes
    //     if (isset($changes['status']) && $changes['status'] !== $original['status']) {
    //         // $this->handleStatusChange($tournament, $original['status'], $changes['status']);
    //     }

    //     // Handle registration status changes
    //     if (isset($changes['registration_status']) && $changes['registration_status'] !== $original['registration_status']) {
    //         // $this->handleRegistrationStatusChange($tournament, $original['registration_status'], $changes['registration_status']);
    //     }

    //     // Handle publication
    //     if (isset($changes['published_at']) && !$original['published_at'] && $changes['published_at']) {
    //         // $this->notifyAllUsers(new TournamentPublishedNotification($tournament));
    //         // $this->notifyAdmins(new TournamentPublishedAdminNotification($tournament));
    //     }

    //     // Handle completion
    //     if (isset($changes['completed_at']) && !$original['completed_at'] && $changes['completed_at']) {
    //         // $this->handleTournamentCompletion($tournament);
    //     }

    //     // Handle cancellation
    //     if (isset($changes['cancelled_at']) && !$original['cancelled_at'] && $changes['cancelled_at']) {
    //         // $this->handleTournamentCancellation($tournament);
    //     }

    //     // Handle basic tournament updates (name, description, dates, etc.)
    //     if ($this->hasBasicTournamentChanges($changes)) {
    //         // $this->notifyTournamentParticipants($tournament, new TournamentUpdatedNotification($tournament, $changes));
    //         // $this->notifyAdmins(new TournamentUpdatedAdminNotification($tournament, $changes));
    //     }

    //     // Handle prize pool changes
    //     if (isset($changes['prize_pool']) && $changes['prize_pool'] !== $original['prize_pool']) {
    //         // $this->notifyTournamentParticipants($tournament, new \App\Notifications\User\TournamentPrizePoolUpdatedNotification($tournament, $original['prize_pool'], $changes['prize_pool']));
    //     }

    //     // Handle entry fee changes
    //     if (isset($changes['entry_fee']) && $changes['entry_fee'] !== $original['entry_fee']) {
    //         // $this->notifyTournamentParticipants($tournament, new \App\Notifications\User\TournamentEntryFeeUpdatedNotification($tournament, $original['entry_fee'], $changes['entry_fee']));
    //     }

    //     // Handle streaming status changes
    //     if (isset($changes['is_streamed']) && $changes['is_streamed'] !== $original['is_streamed']) {
    //         if ($changes['is_streamed']) {
    //             // $this->notifyAllUsers(new \App\Notifications\User\TournamentStreamingAnnouncedNotification($tournament));
    //         }
    //     }

    //     // Handle featured status changes
    //     if (isset($changes['is_featured']) && $changes['is_featured'] !== $original['is_featured']) {
    //         if ($changes['is_featured']) {
    //             // $this->notifyAllUsers(new \App\Notifications\User\TournamentFeaturedNotification($tournament));
    //         }
    //     }

    //     // Handle registration date changes
    //     if (isset($changes['registration_start']) || isset($changes['registration_end'])) {
    //         // $this->notifyTournamentParticipants($tournament, new \App\Notifications\User\TournamentRegistrationDatesChangedNotification($tournament, $changes));
    //     }

    //     // Handle tournament date changes
    //     if (isset($changes['start_date']) || isset($changes['end_date'])) {
    //         // $this->notifyTournamentParticipants($tournament, new \App\Notifications\User\TournamentDatesChangedNotification($tournament, $changes));
    //     }

    //     // Handle venue/location changes
    //     if (isset($changes['location']) || isset($changes['venue'])) {
    //         // $this->notifyTournamentParticipants($tournament, new \App\Notifications\User\TournamentVenueChangedNotification($tournament, $changes));
    //     }

    //     // Handle rule changes
    //     if (isset($changes['rules'])) {
    //         // $this->notifyTournamentParticipants($tournament, new \App\Notifications\User\TournamentRulesUpdatedNotification($tournament));
    //     }

    //     // Log the event
    //     \Log::info('Tournament updated', [
    //         'tournament_id' => $tournament->id,
    //         'name' => $tournament->name,
    //         'changes' => array_keys($changes),
    //         'updated_at' => $tournament->updated_at,
    //     ]);
    // }

    /**
     * Handle the Tournament "deleted" event.
     */
    // public function deleted(Tournament $tournament): void
    // {
    //     // Notify all tournament participants
    //     // $this->notifyTournamentParticipants($tournament, new \App\Notifications\User\TournamentDeletedNotification($tournament));

    //     // Notify admins
    //     // $this->notifyAdmins(new \App\Notifications\Admin\TournamentDeletedAdminNotification($tournament));

    //     // Log the event
    //     \Log::info('Tournament deleted', [
    //         'tournament_id' => $tournament->id,
    //         'name' => $tournament->name,
    //         'deleted_at' => now(),
    //     ]);
    // }

    /**
     * Handle the Tournament "restored" event.
     */
    public function restored(Tournament $tournament): void
    {
        // Notify all tournament participants
        $this->notifyTournamentParticipants($tournament, new \App\Notifications\User\TournamentRestoredNotification($tournament));

        // Notify admins
        $this->notifyAdmins(new \App\Notifications\Admin\TournamentRestoredAdminNotification($tournament));

        // Log the event
        \Log::info('Tournament restored', [
            'tournament_id' => $tournament->id,
            'name' => $tournament->name,
            'restored_at' => now(),
        ]);
    }

    /**
     * Handle tournament status changes.
     */
    private function handleStatusChange(Tournament $tournament, $oldStatus, $newStatus): void
    {
        // Notify participants
        $this->notifyTournamentParticipants($tournament, new TournamentStatusChangedNotification($tournament, $oldStatus, $newStatus));

        // Notify admins
        $this->notifyAdmins(new TournamentStatusChangedAdminNotification($tournament, $oldStatus, $newStatus));

        // Handle specific status changes
        switch ($newStatus) {
            case 'active':
                $this->notifyTournamentParticipants($tournament, new TournamentStartedNotification($tournament));
                $this->notifyAdmins(new TournamentStartedAdminNotification($tournament));
                break;

            case 'completed':
                $this->handleTournamentCompletion($tournament);
                break;

            case 'cancelled':
                $this->handleTournamentCancellation($tournament);
                break;
        }
    }

    /**
     * Handle tournament registration status changes.
     */
    private function handleRegistrationStatusChange(Tournament $tournament, $oldStatus, $newStatus): void
    {
        switch ($newStatus) {
            case 'open':
                $this->notifyAllUsers(new TournamentRegistrationOpenedNotification($tournament));
                $this->notifyAdmins(new TournamentRegistrationOpenedAdminNotification($tournament));
                break;

            case 'closed':
                $this->notifyTournamentParticipants($tournament, new TournamentRegistrationClosedNotification($tournament));
                $this->notifyAdmins(new TournamentRegistrationClosedAdminNotification($tournament));
                break;
        }
    }

    /**
     * Handle tournament completion.
     */
    private function handleTournamentCompletion(Tournament $tournament): void
    {
        // Notify all participants
        $this->notifyTournamentParticipants($tournament, new TournamentCompletedNotification($tournament));

        // Notify admins
        $this->notifyAdmins(new TournamentCompletedAdminNotification($tournament));

        // Notify all users about tournament results if it's a featured tournament
        if ($tournament->is_featured) {
            $this->notifyAllUsers(new \App\Notifications\User\FeaturedTournamentCompletedNotification($tournament));
        }

        // Send certificates to participants if applicable
        $this->sendParticipationCertificates($tournament);
    }

    /**
     * Handle tournament cancellation.
     */
    private function handleTournamentCancellation(Tournament $tournament): void
    {
        // Notify all participants
        $this->notifyTournamentParticipants($tournament, new TournamentCancelledNotification($tournament));

        // Notify admins
        $this->notifyAdmins(new TournamentCancelledAdminNotification($tournament));

        // Handle refunds if applicable
        $this->handleTournamentRefunds($tournament);
    }

    /**
     * Send participation certificates to all tournament participants.
     */
    private function sendParticipationCertificates(Tournament $tournament): void
    {
        // Get all teams that participated
        $teams = $tournament->teams()->wherePivot('status', 'active')->get();
        
        foreach ($teams as $team) {
            // Send certificate notification to team captain
            if ($team->captain) {
                $team->captain->notify(new \App\Notifications\User\ParticipationCertificateAvailableNotification($tournament, $team));
            }
        }
    }

    /**
     * Handle tournament refunds.
     */
    private function handleTournamentRefunds(Tournament $tournament): void
    {
        // This would integrate with payment processing system
        // For now, just notify about refund eligibility
        $this->notifyTournamentParticipants($tournament, new \App\Notifications\User\TournamentRefundEligibleNotification($tournament));
    }

    /**
     * Check if the changes include basic tournament information fields.
     */
    private function hasBasicTournamentChanges(array $changes): bool
    {
        $basicFields = ['name', 'description', 'location', 'venue', 'contact_email', 'contact_phone', 'website'];
        
        foreach ($basicFields as $field) {
            if (isset($changes[$field])) {
                return true;
            }
        }
        
        return false;
    }

    /**
     * Notify all tournament participants (players and teams).
     */
    private function notifyTournamentParticipants(Tournament $tournament, $notification): void
    {
        $participants = collect();

        // Get all players in this tournament
        $players = $tournament->players()->with('user')->get();
        foreach ($players as $player) {
            if ($player->user) {
                $participants->push($player->user);
            }
        }

        // Get all teams and their members
        $teams = $tournament->teams()->with(['captain', 'members'])->get();
        foreach ($teams as $team) {
            if ($team->captain) {
                $participants->push($team->captain);
            }
            if ($team->members) {
                $participants = $participants->merge($team->members);
            }
        }

        // Remove duplicates and send notifications
        $participants = $participants->unique('id');
        if ($participants->isNotEmpty()) {
            Notification::send($participants, $notification);
        }
    }

    /**
     * Notify all users in the system.
     */
    private function notifyAllUsers($notification): void
    {
        // For performance, you might want to queue this or limit to active users
        $users = User::where('created_at', '>=', now()->subMonths(6))->get();
        
        if ($users->isNotEmpty()) {
            Notification::send($users, $notification);
        }
    }

    /**
     * Notify all admin users.
     */
    private function notifyAdmins($notification): void
    {
       // Notify both admin and super_admin users
       $admins = User::whereHas('roles', function ($query) {
                $query->whereIn('name', ['admin', 'super_admin']);
            })->get();

        Notification::send($admins, $notification);
    }
}