<?php

namespace App\Observers;

use App\Models\TeamOfficial;
use App\Models\User;
use App\Notifications\User\{
    OfficialAssignmentNotification,
    OfficialOtpNotification,
    OfficialRoleChangedNotification,
    OfficialEmailChangedNotification,
    OfficialEmailChangedOldNotification,
    OfficialQualificationsUpdatedNotification,
    OfficialVerifiedNotification,
    OfficialRemovedNotification,
    OfficialRestoredNotification,
    TeamOfficialAssignedNotification,
    TeamOfficialRoleChangedNotification,
    TeamOfficialQualificationsUpdatedNotification,
    TeamOfficialVerifiedNotification,
    TeamOfficialRemovedNotification,
    TeamOfficialRestoredNotification
};
use App\Notifications\Admin\{
    OfficialAssignedAdminNotification,
    OfficialRoleChangedAdminNotification,
    OfficialVerificationUpdatedAdminNotification,
    OfficialQualificationsUpdatedAdminNotification,
    OfficialOtpGeneratedAdminNotification,
    TeamOfficialAssignedAdminNotification,
    TeamOfficialRoleChangedAdminNotification,
    TeamOfficialQualificationsUpdatedAdminNotification,
    TeamOfficialVerifiedAdminNotification,
    TeamOfficialRemovedAdminNotification,
    TeamOfficialRestoredAdminNotification
};

use Illuminate\Support\Facades\Notification;
use Illuminate\Support\Facades\Mail;

class TeamOfficialObserver
{
    /**
     * Handle the TeamOfficial "created" event.
     */
    public function created(TeamOfficial $official): void
    {
        // Load relationships
        $official->load(['team']);

        // Send notification to the official about their assignment
        if ($official->email) {
            $official->notify(new OfficialAssignmentNotification($official));
        }

        // Notify team captain about new official assignment
        if ($official->team && $official->team->captain) {
            $official->team->captain->notify(new TeamOfficialAssignedNotification($official->team->id, $official->team->name, $official->role));
        }

        // Notify team members about new official
        $this->notifyTeamMembers($official->team, $official->id, $official->name, $official->role);

        // Notify admins about new official assignment
        $this->notifyAdmins(new OfficialAssignedAdminNotification($official));

        // Generate OTP for verification if email is provided
        if ($official->email) {
            $otp = $official->generateOtp();
            
            // Send OTP via notification
            $official->notify(new OfficialOtpNotification($official, $otp));
            
            // Notify admins about OTP generation
            $this->notifyAdmins(new OfficialOtpGeneratedAdminNotification($official));
        }

    }

    /**
     * Handle the TeamOfficial "updated" event.
     */
    public function updated(TeamOfficial $official): void
    {
        $changes = $official->getChanges();
        $original = $official->getOriginal();
        $official->load(['team']);

        // Handle role changes
        if (isset($changes['role']) && isset($original['role']) && $changes['role'] !== $original['role'] && $official->team) {
            // Send notification to official about role change
            if ($official->email) {
                $official->notify(new OfficialRoleChangedNotification($official, $original['role'], $changes['role']));
            }

            // Notify team captain
            if ($official->team && $official->team->captain) {
                $official->team->captain->notify(new TeamOfficialRoleChangedNotification($official, $original['role'], $changes['role']));
            }

            // Notify team members
            $this->notifyTeamMembers($official->team, new TeamOfficialRoleChangedNotification($official, $original['role'], $changes['role']));

            // Notify admins
            $this->notifyAdmins(new OfficialRoleChangedAdminNotification($official, $original['role'], $changes['role']));
        }

        // Handle email changes
        if (isset($changes['email']) && isset($original['email']) && $changes['email'] !== $original['email']) {
            // Send confirmation to new email
            if ($changes['email']) {
                $official->notify(new OfficialEmailChangedNotification($official, $original['email']));
            }

            // Send notification to old email
            if ($original['email']) {
                Mail::to($original['email'])->send(new OfficialEmailChangedOldNotification($official, $changes['email']));
            }

            // Generate new OTP for new email verification
            if ($changes['email']) {
                $otp = $official->generateOtp();
                $official->notify(new OfficialOtpNotification($official, $otp));
            }
        }

        // Handle qualifications changes
        if (isset($changes['qualifications']) && isset($original['qualifications']) && $changes['qualifications'] !== $original['qualifications']) {
            // Send notification to official
            if ($official->email) {
                $official->notify(new OfficialQualificationsUpdatedNotification($official));
            }

            // Notify team captain
            if ($official->team && $official->team->captain) {
                $official->team->captain->notify(new TeamOfficialQualificationsUpdatedNotification($official));
            }

            // Notify admins
            $this->notifyAdmins(new OfficialQualificationsUpdatedAdminNotification($official, $original['qualifications'], $changes['qualifications']));
        }

        // Handle verification status changes
        if (isset($changes['otp_verified']) && isset($original['otp_verified']) && $changes['otp_verified'] !== $original['otp_verified']) {
            if ($changes['otp_verified']) {
                // Official verified successfully
                if ($official->email) {
                            $official->notify(new OfficialVerifiedNotification($official));
                }

                // Notify team captain
                if ($official->team && $official->team->captain) {
                    $official->team->captain->notify(new TeamOfficialVerifiedNotification($official));
                }

                // Notify team members
                $this->notifyTeamMembers($official->team, new TeamOfficialVerifiedNotification($official));

                // Notify admins
                $this->notifyAdmins(new OfficialVerificationUpdatedAdminNotification($official, false, true));
            }
        }

        // Handle phone changes
        if (isset($changes['phone']) && isset($original['phone']) && $changes['phone'] !== $original['phone']) {
            // Send SMS notification if phone number is provided
            if ($changes['phone']) {
                // Here you would integrate with SMS service
                // SMS::to($changes['phone'])->send(new OfficialPhoneUpdatedSms($official));
            }
        }


    }

    /**
     * Handle the TeamOfficial "deleted" event.
     */
    public function deleted(TeamOfficial $official): void
    {
        $official->load(['team']);

        // Send notification to the official
        if ($official->email) {
            $official->notify(new OfficialRemovedNotification($official));
        }

        // Notify team captain
        if ($official->team && $official->team->captain) {
            $official->team->captain->notify(new TeamOfficialRemovedNotification($official));
        }

        // Notify team members
        $this->notifyTeamMembers($official->team, new TeamOfficialRemovedNotification($official));

        // Notify admins
        $this->notifyAdmins(new OfficialRemovedAdminNotification($official));


    }

    /**
     * Handle the TeamOfficial "restored" event.
     */
    public function restored(TeamOfficial $official): void
    {
        $official->load(['team']);

        // Send notification to the official
        if ($official->email) {
            $official->notify(new OfficialRestoredNotification($official));
        }

        // Notify team captain
        if ($official->team && $official->team->captain) {
            $official->team->captain->notify(new TeamOfficialRestoredNotification($official));
        }

        // Notify team members
        $this->notifyTeamMembers($official->team, new TeamOfficialRestoredNotification($official));

        // Notify admins
        $this->notifyAdmins(new OfficialRestoredAdminNotification($official));


    }

    /**
     * Notify all team members.
     */
    private function notifyTeamMembers($team, $notification): void
    {
        if (!$team) return;

        $members = collect();

        // Add team members
        if ($team->members) {
            $members = $members->merge($team->members);
        }

        if ($members->isNotEmpty()) {
            Notification::send($members, $notification);
        }
    }

    /**
     * Notify all admin users.
     */
    private function notifyAdmins($notification): void
    {
        // Notify both admin and super_admin users
        $admins = User::whereHas('roles', function ($query) {
            $query->whereIn('name', ['admin', 'super_admin', 'organizer']);
        })->get();

        Notification::send($admins, $notification);
    }
}