<?php

namespace App\Observers;

use App\Models\Team;
use App\Models\User;
use App\Notifications\User\TeamCreatedNotification;
use App\Notifications\User\TeamUpdatedNotification;
use App\Notifications\User\TeamTournamentRegisteredNotification;
use App\Notifications\User\TeamStatusChangedNotification;
use App\Notifications\User\TeamPaymentStatusUpdatedNotification;
use App\Notifications\User\TeamMemberAddedNotification;
use App\Notifications\User\TeamMemberWelcomeNotification;
use App\Notifications\User\TeamMemberRemovedNotification;
use App\Notifications\Admin\TeamCreatedAdminNotification;
use App\Notifications\Admin\TeamUpdatedAdminNotification;
use App\Notifications\Admin\TeamTournamentRegisteredAdminNotification;
use App\Notifications\Admin\TeamStatusChangedAdminNotification;
use App\Notifications\Admin\TeamPaymentStatusUpdatedAdminNotification;
use App\Notifications\Admin\TeamMemberChangedAdminNotification;
use Illuminate\Support\Facades\Notification;

class TeamObserver
{
    /**
     * Handle the Team "created" event.
     */
    public function created(Team $team): void
    {
        // Load relationships
        $team->load(['captain', 'members']);

        // Notify the team captain
        if ($team->captain) {
            $team->captain->notify(new TeamCreatedNotification($team));
        }

        // Notify all team members
        $this->notifyTeamMembers($team, new TeamMemberWelcomeNotification($team->id, $team->name));

        // Notify admins and organizers
        $this->notifyAdmins(new TeamCreatedAdminNotification($team));

        // Log the event
        \Log::info('Team created', [
            'team_id' => $team->id,
            'name' => $team->name,
            'captain_id' => $team->captain_id,
            'status' => $team->status,
            'created_at' => $team->created_at,
        ]);
    }

    /**
     * Handle the Team "updated" event.
     */
    public function updated(Team $team): void
    {
        $changes = $team->getChanges();
        $original = $team->getOriginal();
        $team->load(['captain', 'members']);

        // Handle basic team updates (name, description, logo)
        if ($this->hasBasicTeamChanges($changes)) {
            $this->notifyTeamMembers($team, new TeamUpdatedNotification($team, $changes));
            $this->notifyAdmins(new TeamUpdatedAdminNotification($team, $changes));
        }

        // Handle status changes
        if (isset($changes['status']) && $changes['status'] !== $original['status']) {
            $this->notifyTeamMembers($team, new TeamStatusChangedNotification($team, $original['status'], $changes['status']));
            $this->notifyAdmins(new TeamStatusChangedAdminNotification($team, $original['status'], $changes['status']));
        }

        // Handle payment status changes
        if (isset($changes['payment_status']) && $changes['payment_status'] !== $original['payment_status']) {
            $this->notifyTeamMembers($team, new TeamPaymentStatusUpdatedNotification($team, $original['payment_status'], $changes['payment_status']));
            $this->notifyAdmins(new TeamPaymentStatusUpdatedAdminNotification($team, $original['payment_status'], $changes['payment_status']));
        }

        // Handle captain changes
        if (isset($changes['captain_id']) && $changes['captain_id'] !== $original['captain_id']) {
            // Notify old captain
            if ($original['captain_id']) {
                $oldCaptain = User::find($original['captain_id']);
                if ($oldCaptain) {
                    $oldCaptain->notify(new TeamCaptainRemovedNotification($team));
                }
            }

            // Notify new captain
            if ($team->captain) {
                $team->captain->notify(new TeamCaptainAssignedNotification($team));
            }

            // Notify all team members
            $this->notifyTeamMembers($team, new TeamCaptainChangedNotification($team, $original['captain_id'], $changes['captain_id']));

            // Notify admins
            $this->notifyAdmins(new TeamCaptainChangedAdminNotification($team, $original['captain_id'], $changes['captain_id']));
        }

        // Handle payment completion
        if (isset($changes['payment_completed_at']) && !$original['payment_completed_at'] && $changes['payment_completed_at']) {
            $this->notifyTeamMembers($team, new \App\Notifications\User\TeamPaymentCompletedNotification($team));
            $this->notifyAdmins(new \App\Notifications\Admin\TeamPaymentCompletedAdminNotification($team));
        }

        // Handle max players changes
        if (isset($changes['max_players']) && $changes['max_players'] !== $original['max_players']) {
            $this->notifyTeamMembers($team, new \App\Notifications\User\TeamMaxPlayersChangedNotification($team, $original['max_players'], $changes['max_players']));
        }

        // Log the event
        \Log::info('Team updated', [
            'team_id' => $team->id,
            'name' => $team->name,
            'changes' => array_keys($changes),
            'updated_at' => $team->updated_at,
        ]);
    }

    /**
     * Handle the Team "deleted" event.
     */
    public function deleted(Team $team): void
    {
        $team->load(['captain', 'members']);

        // Notify team captain
        if ($team->captain) {
            $team->captain->notify(new \App\Notifications\User\TeamDeletedNotification($team));
        }

        // Notify all team members
        $this->notifyTeamMembers($team, new \App\Notifications\User\TeamDeletedMemberNotification($team));

        // Notify admins
        $this->notifyAdmins(new \App\Notifications\Admin\TeamDeletedAdminNotification($team));

        // Log the event
        \Log::info('Team deleted', [
            'team_id' => $team->id,
            'name' => $team->name,
            'captain_id' => $team->captain_id,
            'deleted_at' => now(),
        ]);
    }

    /**
     * Handle the Team "restored" event.
     */
    public function restored(Team $team): void
    {
        $team->load(['captain', 'members']);

        // Notify team captain
        if ($team->captain) {
            $team->captain->notify(new \App\Notifications\User\TeamRestoredNotification($team));
        }

        // Notify all team members
        $this->notifyTeamMembers($team, new \App\Notifications\User\TeamRestoredMemberNotification($team));

        // Notify admins
        $this->notifyAdmins(new \App\Notifications\Admin\TeamRestoredAdminNotification($team));

        // Log the event
        \Log::info('Team restored', [
            'team_id' => $team->id,
            'name' => $team->name,
            'restored_at' => now(),
        ]);
    }

    /**
     * Handle team member pivot events.
     */
    public function memberAttached(Team $team, $memberId, array $attributes): void
    {
        $member = User::find($memberId);
        if (!$member) return;

        $team->load(['captain']);

        // Notify the new member
        $member->notify(new TeamMemberAddedNotification($team, $attributes['role'] ?? 'member'));

        // Notify team captain
        if ($team->captain && $team->captain->id !== $memberId) {
            $team->captain->notify(new \App\Notifications\User\TeamNewMemberNotification($team, $member, $attributes['role'] ?? 'member'));
        }

        // Notify other team members
        $otherMembers = $team->members()->where('users.id', '!=', $memberId)->get();
        Notification::send($otherMembers, new \App\Notifications\User\TeamNewMemberNotification($team, $member, $attributes['role'] ?? 'member'));

        // Notify admins
        $this->notifyAdmins(new TeamMemberChangedAdminNotification($team, $member, 'added', $attributes['role'] ?? 'member'));

        // Log the event
        \Log::info('Team member added', [
            'team_id' => $team->id,
            'member_id' => $memberId,
            'role' => $attributes['role'] ?? 'member',
            'added_at' => now(),
        ]);
    }

    /**
     * Handle team member pivot events.
     */
    public function memberDetached(Team $team, $memberId): void
    {
        $member = User::find($memberId);
        if (!$member) return;

        $team->load(['captain']);

        // Notify the removed member
        $member->notify(new TeamMemberRemovedNotification($team));

        // Notify team captain
        if ($team->captain && $team->captain->id !== $memberId) {
            $team->captain->notify(new \App\Notifications\User\TeamMemberLeftNotification($team, $member));
        }

        // Notify remaining team members
        $remainingMembers = $team->members()->where('users.id', '!=', $memberId)->get();
        Notification::send($remainingMembers, new \App\Notifications\User\TeamMemberLeftNotification($team, $member));

        // Notify admins
        $this->notifyAdmins(new TeamMemberChangedAdminNotification($team, $member, 'removed'));

        // Log the event
        \Log::info('Team member removed', [
            'team_id' => $team->id,
            'member_id' => $memberId,
            'removed_at' => now(),
        ]);
    }

    /**
     * Check if the changes include basic team information fields.
     */
    private function hasBasicTeamChanges(array $changes): bool
    {
        $basicFields = ['name', 'description', 'logo'];
        
        foreach ($basicFields as $field) {
            if (isset($changes[$field])) {
                return true;
            }
        }
        
        return false;
    }

    /**
     * Notify all team members.
     */
    private function notifyTeamMembers(Team $team, $notification): void
    {
        $members = collect();

        // Add captain
        if ($team->captain) {
            $members->push($team->captain);
        }

        // Add team members
        $teamMembers = $team->members ?? collect();
        $members = $members->merge($teamMembers);

        // Remove duplicates
        $members = $members->unique('id');

        if ($members->isNotEmpty()) {
            Notification::send($members, $notification);
        }
    }

    /**
     * Notify all admin users.
     */
    private function notifyAdmins($notification): void
    {
        // Notify both admin and super_admin users
        $admins = User::whereHas('roles', function ($query) {
            $query->whereIn('name', ['admin', 'super_admin', 'organizer']);
        })->get();

        Notification::send($admins, $notification);
    }
}