<?php

namespace App\Observers;

use App\Models\Player;
use App\Models\User;
use App\Notifications\User\PlayerRegisteredNotification;
use App\Notifications\User\PlayerStatusChangedNotification;
use App\Notifications\User\PlayerVerificationUpdatedNotification;
use App\Notifications\User\PlayerTournamentRegisteredNotification;
use App\Notifications\User\PlayerRatingUpdatedNotification;
use App\Notifications\User\PlayerDigitalIdGeneratedNotification;
use App\Notifications\Admin\PlayerRegisteredAdminNotification;
use App\Notifications\Admin\PlayerStatusChangedAdminNotification;
use App\Notifications\Admin\PlayerVerificationUpdatedAdminNotification;
use App\Notifications\Admin\PlayerTournamentRegisteredAdminNotification;
use App\Notifications\Admin\PlayerRatingUpdatedAdminNotification;
use App\Notifications\Admin\PlayerDigitalIdGeneratedAdminNotification;
use Illuminate\Support\Facades\Notification;
use Barryvdh\DomPDF\Facade\Pdf;

class PlayerObserver
{
    /**
     * Handle the Player "created" event.
     */
    public function created(Player $player): void
    {
        // Load relationships
        $player->load(['user', 'tournament', 'team']);

        // Notify the player's user about successful registration
        $player->user->notify(new PlayerRegisteredNotification($player));

        // Notify tournament admins and organizers
        $this->notifyTournamentStaff($player->tournament, new PlayerRegisteredAdminNotification($player));

        // Notify team captain if player is part of a team
        if ($player->team && $player->team->captain) {
            $player->team->captain->notify(new \App\Notifications\User\TeamPlayerAddedNotification($player));
        }

        // Log the event
        \Log::info('Player created', [
            'player_id' => $player->id,
            'user_id' => $player->user_id,
            'tournament_id' => $player->tournament_id,
            'team_id' => $player->team_id,
            'display_name' => $player->display_name,
            'player_type' => $player->player_type,
            'status' => $player->status,
            'created_at' => $player->created_at,
        ]);
    }

    /**
     * Handle the Player "updated" event.
     */
    public function updated(Player $player): void
    {
        $changes = $player->getChanges();
        $original = $player->getOriginal();
        $player->load(['user', 'tournament', 'team']);

        // Handle status changes
        if (isset($changes['status']) && $changes['status'] !== $original['status']) {
            $player->user->notify(new PlayerStatusChangedNotification($player, $original['status'], $changes['status']));
            $this->notifyTournamentStaff($player->tournament, new PlayerStatusChangedAdminNotification($player, $original['status'], $changes['status']));
        }

        // Handle verification status changes
        if (isset($changes['verification_status']) && $changes['verification_status'] !== $original['verification_status']) {
            $player->user->notify(new PlayerVerificationUpdatedNotification($player, $original['verification_status'], $changes['verification_status']));
            $this->notifyTournamentStaff($player->tournament, new PlayerVerificationUpdatedAdminNotification($player, $original['verification_status'], $changes['verification_status']));
        }

        // Handle tournament changes (player switching tournaments)
        if (isset($changes['tournament_id']) && $changes['tournament_id'] !== $original['tournament_id']) {
            $player->user->notify(new PlayerTournamentRegisteredNotification($player));
            $this->notifyTournamentStaff($player->tournament, new PlayerTournamentRegisteredAdminNotification($player));
        }

        // Handle rating updates
        if ($this->hasRatingChanges($changes)) {
            $player->user->notify(new PlayerRatingUpdatedNotification($player, $changes));
            // $this->notifyTournamentStaff($player->tournament, new PlayerRatingUpdatedAdminNotification($player, $changes));
        }

        // Handle digital ID generation
        if (isset($changes['digital_id']) && !$original['digital_id'] && $changes['digital_id']) {
            try {
                // Generate PDF for email attachment
                $data = [
                    'player' => $player,
                    'digital_id' => $player->digital_id,
                    'generated_at' => now()->format('F j, Y'),
                    'valid_until' => now()->addYears(2)->format('F j, Y'),
                    'qr_code' => $this->generateQRCode($player->digital_id),
                ];

                $pdf = Pdf::loadView('players.digital-id-pdf', $data);
                $pdf->setOptions([
                    'isHtml5ParserEnabled' => true,
                    'isRemoteEnabled' => true,
                ]);
                $pdf->getDomPDF()->setHttpContext(stream_context_create([
                    'ssl' => [
                        'verify_peer' => false,
                        'verify_peer_name' => false,
                        'allow_self_signed' => true,
                    ],
                ]));
                $pdf->setPaper('a4', 'landscape');
                
                $pdfContent = $pdf->output();
                
                // Send notifications with PDF attachment
                $player->user->notify(new PlayerDigitalIdGeneratedNotification($player, $pdfContent));
                $this->notifyTournamentStaff($player->tournament, new PlayerDigitalIdGeneratedAdminNotification($player, $pdfContent));
                
            } catch (\Exception $e) {
                \Log::error('Failed to generate PDF for digital ID notification', [
                    'player_id' => $player->id,
                    'error' => $e->getMessage()
                ]);
                
                // Send notifications without PDF attachment as fallback
                $player->user->notify(new PlayerDigitalIdGeneratedNotification($player));
                $this->notifyTournamentStaff($player->tournament, new PlayerDigitalIdGeneratedAdminNotification($player));
            }
        }

        // Handle team changes
        if (isset($changes['team_id']) && $changes['team_id'] !== $original['team_id']) {
            // Notify old team captain
            if ($original['team_id']) {
                $oldTeam = \App\Models\Team::find($original['team_id']);
                if ($oldTeam && $oldTeam->captain) {
                    $oldTeam->captain->notify(new \App\Notifications\User\TeamPlayerRemovedNotification($player, $oldTeam));
                }
            }

            // Notify new team captain
            if ($player->team && $player->team->captain) {
                $player->team->captain->notify(new \App\Notifications\User\TeamPlayerAddedNotification($player));
            }
        }

        // Handle captain status changes
        if (isset($changes['is_team_captain']) && $changes['is_team_captain'] !== $original['is_team_captain']) {
            $player->user->notify(new \App\Notifications\User\PlayerCaptainStatusChangedNotification($player, $changes['is_team_captain']));
            
            if ($player->team) {
                // $this->notifyTournamentStaff($player->tournament, new \App\Notifications\Admin\PlayerCaptainStatusChangedAdminNotification($player, $changes['is_team_captain']));
            }
        }

        // Log the event
        \Log::info('Player updated', [
            'player_id' => $player->id,
            'user_id' => $player->user_id,
            'changes' => array_keys($changes),
            'updated_at' => $player->updated_at,
        ]);
    }

    /**
     * Handle the Player "deleted" event.
     */
    public function deleted(Player $player): void
    {
        $player->load(['user', 'tournament', 'team']);

        // Notify the user
        $player->user->notify(new \App\Notifications\User\PlayerDeletedNotification($player));

        // Notify tournament staff
        $this->notifyTournamentStaff($player->tournament, new \App\Notifications\Admin\PlayerDeletedAdminNotification($player));

        // Notify team captain if applicable
        if ($player->team && $player->team->captain) {
            $player->team->captain->notify(new \App\Notifications\User\TeamPlayerRemovedNotification($player, $player->team));
        }

        // Log the event
        \Log::info('Player deleted', [
            'player_id' => $player->id,
            'user_id' => $player->user_id,
            'tournament_id' => $player->tournament_id,
            'team_id' => $player->team_id,
            'display_name' => $player->display_name,
            'deleted_at' => now(),
        ]);
    }

    /**
     * Handle the Player "restored" event.
     */
    public function restored(Player $player): void
    {
        $player->load(['user', 'tournament', 'team']);

        // Notify the user
        $player->user->notify(new \App\Notifications\User\PlayerRestoredNotification($player));

        // Notify tournament staff
        $this->notifyTournamentStaff($player->tournament, new \App\Notifications\Admin\PlayerRestoredAdminNotification($player));

        // Log the event
        \Log::info('Player restored', [
            'player_id' => $player->id,
            'user_id' => $player->user_id,
            'restored_at' => now(),
        ]);
    }

    /**
     * Check if the changes include rating-related fields.
     */
    private function hasRatingChanges(array $changes): bool
    {
        $ratingFields = ['global_rating', 'global_rank', 'tournament_rating', 'tournament_rank', 'seeding_position'];
        
        foreach ($ratingFields as $field) {
            if (isset($changes[$field])) {
                return true;
            }
        }
        
        return false;
    }

    /**
     * Notify tournament staff (admins, organizers, referees).
     */
    private function notifyTournamentStaff($tournament, $notification): void
    {
        if (!$tournament) return;

        // Get tournament admins and organizers
        $admins = User::whereHas('roles', function ($query) {
            $query->whereIn('name', ['admin', 'super_admin', 'organizer']);
        })->get();

        // Get tournament-specific staff if available
        // This could be extended to include tournament-specific roles

        Notification::send($admins, $notification);
    }

    /**
     * Generate QR Code for Digital ID
     */
    private function generateQRCode($digitalId)
    {
        // QR code generation using QR Server API (more reliable than Google Charts)
        $qrUrl = 'https://api.qrserver.com/v1/create-qr-code/?size=150x150&data=' . urlencode($digitalId) . '&format=png&margin=2';
        return $qrUrl;
    }
}