<?php

namespace App\Observers;

use App\Models\Payment;
use App\Models\User;
use App\Notifications\User\PaymentInitiatedNotification;
use App\Notifications\User\PaymentCompletedNotification;
use App\Notifications\User\PaymentFailedNotification;
use App\Notifications\User\PaymentProcessingNotification;
use App\Notifications\User\PaymentRefundedNotification;
use App\Notifications\Admin\PaymentInitiatedAdminNotification;
use App\Notifications\Admin\PaymentCompletedAdminNotification;
use App\Notifications\Admin\PaymentFailedAdminNotification;
use App\Notifications\Admin\PaymentProcessingAdminNotification;
use App\Notifications\Admin\PaymentRefundedAdminNotification;
use App\Notifications\Admin\PaymentDisputeAdminNotification;
use Illuminate\Support\Facades\Notification;

class PaymentObserver
{
    /**
     * Handle the Payment "created" event.
     */
    public function created(Payment $payment): void
    {
        // Load relationships
        $payment->load(['user', 'team']);

        // Notify the user about payment initiation
        if ($payment->user) {
            $payment->user->notify(new PaymentInitiatedNotification($payment));
        }

        // Notify team managers if payment is for a team
        if ($payment->team) {
            $this->notifyTeamManagers($payment->team, new \App\Notifications\User\TeamPaymentInitiatedNotification($payment));
        }

        // Notify admins about new payment
        $this->notifyAdmins(new PaymentInitiatedAdminNotification($payment));
    }

    /**
     * Handle the Payment "updated" event.
     */
    public function updated(Payment $payment): void
    {
        $changes = $payment->getChanges();
        $original = $payment->getOriginal();
        $payment->load(['user', 'team']);

        // Handle status changes
        if (isset($changes['status']) && $changes['status'] !== $original['status']) {
            $this->handleStatusChange($payment, $original['status'], $changes['status']);
        }

        // Handle payment method changes
        if (isset($changes['payment_method']) && $changes['payment_method'] !== $original['payment_method']) {
            if ($payment->user) {
                $payment->user->notify(new \App\Notifications\User\PaymentMethodChangedNotification($payment, $original['payment_method'], $changes['payment_method']));
            }
            $this->notifyAdmins(new \App\Notifications\Admin\PaymentMethodChangedAdminNotification($payment, $original['payment_method'], $changes['payment_method']));
        }

        // Handle amount changes
        if (isset($changes['amount']) && $changes['amount'] !== $original['amount']) {
            if ($payment->user) {
                $payment->user->notify(new \App\Notifications\User\PaymentAmountChangedNotification($payment, $original['amount'], $changes['amount']));
            }
            $this->notifyAdmins(new \App\Notifications\Admin\PaymentAmountChangedAdminNotification($payment, $original['amount'], $changes['amount']));
        }

        // Handle failure reason updates
        if (isset($changes['failure_reason']) && $changes['failure_reason'] !== $original['failure_reason']) {
            if ($payment->user) {
                $payment->user->notify(new \App\Notifications\User\PaymentFailureReasonUpdatedNotification($payment, $changes['failure_reason']));
            }
            $this->notifyAdmins(new \App\Notifications\Admin\PaymentFailureReasonUpdatedAdminNotification($payment, $changes['failure_reason']));
        }

        // Handle reference number updates
        if (isset($changes['reference_number']) && $changes['reference_number'] !== $original['reference_number']) {
            if ($payment->user) {
                $payment->user->notify(new \App\Notifications\User\PaymentReferenceUpdatedNotification($payment, $original['reference_number'], $changes['reference_number']));
            }
        }

        // Log the event
        \Log::info('Payment updated', [
            'payment_id' => $payment->id,
            'user_id' => $payment->user_id,
            'changes' => array_keys($changes),
            'updated_at' => $payment->updated_at,
        ]);
    }

    /**
     * Handle the Payment "deleted" event.
     */
    public function deleted(Payment $payment): void
    {
        $payment->load(['user', 'team']);

        // Notify the user about payment deletion
        if ($payment->user) {
            $payment->user->notify(new \App\Notifications\User\PaymentDeletedNotification($payment));
        }

        // Notify team managers if applicable
        if ($payment->team) {
            $this->notifyTeamManagers($payment->team, new \App\Notifications\User\TeamPaymentDeletedNotification($payment));
        }

        // Notify admins
        $this->notifyAdmins(new \App\Notifications\Admin\PaymentDeletedAdminNotification($payment));

        // Log the event
        \Log::info('Payment deleted', [
            'payment_id' => $payment->id,
            'user_id' => $payment->user_id,
            'team_id' => $payment->team_id,
            'amount' => $payment->amount,
            'status' => $payment->status,
            'deleted_at' => now(),
        ]);
    }

    /**
     * Handle the Payment "restored" event.
     */
    public function restored(Payment $payment): void
    {
        $payment->load(['user', 'team']);

        // Notify the user about payment restoration
        if ($payment->user) {
            $payment->user->notify(new \App\Notifications\User\PaymentRestoredNotification($payment));
        }

        // Notify team managers if applicable
        if ($payment->team) {
            $this->notifyTeamManagers($payment->team, new \App\Notifications\User\TeamPaymentRestoredNotification($payment));
        }

        // Notify admins
        $this->notifyAdmins(new \App\Notifications\Admin\PaymentRestoredAdminNotification($payment));

        // Log the event
        \Log::info('Payment restored', [
            'payment_id' => $payment->id,
            'user_id' => $payment->user_id,
            'restored_at' => now(),
        ]);
    }

    /**
     * Handle payment status changes.
     */
    private function handleStatusChange(Payment $payment, $oldStatus, $newStatus): void
    {
        switch ($newStatus) {
            case 'processing':
                if ($payment->user) {
                    $payment->user->notify(new PaymentProcessingNotification($payment));
                }
                if ($payment->team) {
                    $this->notifyTeamManagers($payment->team, new \App\Notifications\User\TeamPaymentProcessingNotification($payment));
                }
                $this->notifyAdmins(new PaymentProcessingAdminNotification($payment));
                break;

            case 'completed':
                $this->handlePaymentCompletion($payment);
                break;

            case 'failed':
                if ($payment->user) {
                    $payment->user->notify(new PaymentFailedNotification($payment));
                }
                if ($payment->team) {
                    $this->notifyTeamManagers($payment->team, new \App\Notifications\User\TeamPaymentFailedNotification($payment));
                }
                $this->notifyAdmins(new PaymentFailedAdminNotification($payment));
                break;

            case 'refunded':
                if ($payment->user) {
                    $payment->user->notify(new PaymentRefundedNotification($payment));
                }
                if ($payment->team) {
                    $this->notifyTeamManagers($payment->team, new \App\Notifications\User\TeamPaymentRefundedNotification($payment));
                }
                $this->notifyAdmins(new PaymentRefundedAdminNotification($payment));
                break;

            case 'disputed':
                // Only notify admins for disputes
                $this->notifyAdmins(new PaymentDisputeAdminNotification($payment));
                break;

            case 'cancelled':
                if ($payment->user) {
                    $payment->user->notify(new \App\Notifications\User\PaymentCancelledNotification($payment));
                }
                if ($payment->team) {
                    $this->notifyTeamManagers($payment->team, new \App\Notifications\User\TeamPaymentCancelledNotification($payment));
                }
                $this->notifyAdmins(new \App\Notifications\Admin\PaymentCancelledAdminNotification($payment));
                break;
        }
    }

    /**
     * Handle payment completion.
     */
    private function handlePaymentCompletion(Payment $payment): void
    {
        // Notify the user
        if ($payment->user) {
            $payment->user->notify(new PaymentCompletedNotification($payment));
        }

        // Notify team managers
        if ($payment->team) {
            $this->notifyTeamManagers($payment->team, new \App\Notifications\User\TeamPaymentCompletedNotification($payment));
            
            // Update team payment status if this was a team registration payment
            if ($payment->team->payment_status !== 'completed') {
                $payment->team->update([
                    'payment_status' => 'completed',
                    'payment_completed_at' => now(),
                    'transaction_id' => $payment->reference_number,
                    'payment_amount' => $payment->amount,
                    'payment_method' => $payment->payment_method,
                ]);
            }
        }

        // Notify admins
        $this->notifyAdmins(new PaymentCompletedAdminNotification($payment));

        // Generate receipt
        $this->generatePaymentReceipt($payment);

        // Update user payment status if applicable
        if ($payment->user && $payment->user->payment_status !== 'completed') {
            $payment->user->update(['payment_status' => 'completed']);
        }
    }

    /**
     * Generate payment receipt.
     */
    private function generatePaymentReceipt(Payment $payment): void
    {
        // Send receipt email
        if ($payment->user) {
            $payment->user->notify(new \App\Notifications\User\PaymentReceiptNotification($payment));
        }

        // Log receipt generation
        \Log::info('Payment receipt generated', [
            'payment_id' => $payment->id,
            'user_id' => $payment->user_id,
            'amount' => $payment->amount,
            'reference_number' => $payment->reference_number,
        ]);
    }

    /**
     * Notify all team members.
     */
    private function notifyTeamMembers($team, $notification): void
    {
        if (!$team) return;

        $members = collect();

        // Add captain
        if ($team->captain) {
            $members->push($team->captain);
        }

        // Add team members
        if ($team->members) {
            $members = $members->merge($team->members);
        }

        // Remove duplicates
        $members = $members->unique('id');

        if ($members->isNotEmpty()) {
            Notification::send($members, $notification);
        }
    }

    /**
     * Notify team managers only.
     */
    private function notifyTeamManagers($team, $notification): void
    {
        if (!$team) return;

        $managers = collect();

        // Add team manager (if different from captain)
        if ($team->manager) {
            $managers->push($team->manager);
        }

        // Add captain if they have team_manager role
        if ($team->captain && $team->captain->hasRole('team_manager')) {
            $managers->push($team->captain);
        }

        // Remove duplicates
        $managers = $managers->unique('id');

        if ($managers->isNotEmpty()) {
            Notification::send($managers, $notification);
        }
    }

    /**
     * Notify all admin users.
     */
    private function notifyAdmins($notification): void
    {
        $admins = User::whereHas('roles', function ($query) {
            $query->whereIn('name', ['admin', 'super_admin', 'organizer']);
        })->get();

        Notification::send($admins, $notification);
    }
}