<?php

namespace App\Observers;

use App\Models\GameMatch;
use App\Models\User;
use App\Notifications\User\MatchScheduledNotification;
use App\Notifications\User\MatchStartedNotification;
use App\Notifications\User\MatchCompletedNotification;
use App\Notifications\User\MatchCancelledNotification;
use App\Notifications\User\MatchRescheduledNotification;
use App\Notifications\User\MatchStreamingStartedNotification;
use App\Notifications\User\MatchResultUpdatedNotification;
use App\Notifications\User\MatchVerificationUpdatedNotification;
use App\Notifications\User\MatchRefereeAssignedNotification;
use App\Notifications\Admin\MatchScheduledAdminNotification;
use App\Notifications\Admin\MatchStartedAdminNotification;
use App\Notifications\Admin\MatchCompletedAdminNotification;
use App\Notifications\Admin\MatchCancelledAdminNotification;
use App\Notifications\Admin\MatchRescheduledAdminNotification;
use App\Notifications\Admin\MatchStreamingStartedAdminNotification;
use App\Notifications\Admin\MatchResultUpdatedAdminNotification;
use App\Notifications\Admin\MatchVerificationUpdatedAdminNotification;
use App\Notifications\Admin\MatchRefereeAssignedAdminNotification;
use Illuminate\Support\Facades\Notification;

class GameMatchObserver
{
    /**
     * Handle the GameMatch "created" event.
     */
    public function created(GameMatch $match): void
    {
        // Load relationships
        $match->load(['tournament', 'player1', 'player2', 'team1', 'team2', 'referee']);

        // Notify participants about match creation/scheduling
        // $this->notifyMatchParticipants($match, new MatchScheduledNotification($match));

        // Notify referee if assigned
        if ($match->referee) {
            // $match->referee->notify(new RefereeMatchAssignedNotification($match));
        }

        // Notify tournament staff
        // $this->notifyTournamentStaff($match->tournament, new MatchScheduledAdminNotification($match));

        // Notify spectators if match allows spectators
        if ($match->allow_spectators) {
            // $this->notifyTournamentFollowers($match->tournament, new NewMatchAvailableNotification($match));
        }

        // Log the event
        \Log::info('Match created', [
            'match_id' => $match->id,
            'tournament_id' => $match->tournament_id,
            'match_code' => $match->match_code,
            'match_type' => $match->match_type,
            'scheduled_time' => $match->scheduled_time,
            'status' => $match->status,
            'created_at' => $match->created_at,
        ]);
    }

    /**
     * Handle the GameMatch "updated" event.
     */
    public function updated(GameMatch $match): void
    {
        $changes = $match->getChanges();
        $original = $match->getOriginal();
        $match->load(['tournament', 'player1', 'player2', 'team1', 'team2', 'referee']);

        // Handle status changes
        if (isset($changes['status']) && $changes['status'] !== $original['status']) {
            $this->handleStatusChange($match, $original['status'], $changes['status']);
        }

        // Handle progress status changes
        if (isset($changes['progress_status']) && $changes['progress_status'] !== $original['progress_status']) {
            $this->handleProgressStatusChange($match, $original['progress_status'], $changes['progress_status']);
        }

        // Handle scheduling time changes
        if (isset($changes['scheduled_time']) && $changes['scheduled_time'] !== $original['scheduled_time']) {
            // $this->notifyMatchParticipants($match, new MatchRescheduledNotification($match, $original['scheduled_time'], $changes['scheduled_time']));
            // $this->notifyTournamentStaff($match->tournament, new MatchRescheduledAdminNotification($match, $original['scheduled_time'], $changes['scheduled_time']));
        }

        // Handle streaming status changes
        if (isset($changes['is_streamed']) && $changes['is_streamed'] !== $original['is_streamed']) {
            if ($changes['is_streamed']) {
                // $this->notifyMatchParticipants($match, new MatchStreamingStartedNotification($match));
                // $this->notifyTournamentFollowers($match->tournament, new MatchStreamingNotification($match));
                // $this->notifyTournamentStaff($match->tournament, new MatchStreamingStartedAdminNotification($match));
            }
        }

        // Handle referee assignment changes
        if (isset($changes['referee_id']) && $changes['referee_id'] !== $original['referee_id']) {
            // Notify old referee
            if ($original['referee_id']) {
                $oldReferee = User::find($original['referee_id']);
                if ($oldReferee) {
                    //  $oldReferee->notify(new RefereeMatchUnassignedNotification($match));
                }
            }

            // Notify new referee
            if ($match->referee) {
                // $match->referee->notify(new RefereeMatchAssignedNotification($match));
            }

            // Notify participants
            // $this->notifyMatchParticipants($match, new MatchRefereeAssignedNotification($match));

            // Notify admins
            // $this->notifyTournamentStaff($match->tournament, new MatchRefereeAssignedAdminNotification($match, $original['referee_id'], $changes['referee_id']));
        }

        // Handle result updates
        if ($this->hasResultChanges($changes)) {
            // $this->notifyMatchParticipants($match, new MatchResultUpdatedNotification($match, $changes));
            // $this->notifyTournamentStaff($match->tournament, new MatchResultUpdatedAdminNotification($match, $changes));
        }

        // Handle verification status changes
        if (isset($changes['verification_status']) && $changes['verification_status'] !== $original['verification_status']) {
            // $this->notifyMatchParticipants($match, new MatchVerificationUpdatedNotification($match, $original['verification_status'], $changes['verification_status']));
            // $this->notifyTournamentStaff($match->tournament, new MatchVerificationUpdatedAdminNotification($match, $original['verification_status'], $changes['verification_status']));
        }

        // Handle stream URL updates
        if (isset($changes['stream_url']) && $changes['stream_url'] !== $original['stream_url'] && $changes['stream_url']) {
            // $this->notifyTournamentFollowers($match->tournament, new MatchStreamUrlUpdatedNotification($match));
        }

        // Handle winner determination
        if (isset($changes['winner_id']) && !$original['winner_id'] && $changes['winner_id']) {
            // $this->handleMatchCompletion($match);
        }

        // Handle pause/resume events
        if (isset($changes['paused_at']) && !$original['paused_at'] && $changes['paused_at']) {
            // $this->notifyMatchParticipants($match, new MatchPausedNotification($match));
        }

        if (isset($changes['resumed_at']) && !$original['resumed_at'] && $changes['resumed_at']) {
            // $this->notifyMatchParticipants($match, new MatchResumedNotification($match));
        }

        // Log the event
        \Log::info('Match updated', [
            'match_id' => $match->id,
            'changes' => array_keys($changes),
            'updated_at' => $match->updated_at,
        ]);
    }

    /**
     * Handle the GameMatch "deleted" event.
     */
    public function deleted(GameMatch $match): void
    {
        $match->load(['tournament', 'player1', 'player2', 'team1', 'team2', 'referee']);

        // Notify participants about match deletion
        // $this->notifyMatchParticipants($match, new MatchDeletedNotification($match));

        // Notify referee
        if ($match->referee) {
            // $match->referee->notify(new RefereeMatchDeletedNotification($match));
        }

        // Notify tournament staff
        //  $this->notifyTournamentStaff($match->tournament, new MatchDeletedAdminNotification($match));

        // Log the event
        \Log::info('Match deleted', [
            'match_id' => $match->id,
            'tournament_id' => $match->tournament_id,
            'match_code' => $match->match_code,
            'deleted_at' => now(),
        ]);
    }

    /**
     * Handle match status changes.
     */
    private function handleStatusChange(GameMatch $match, $oldStatus, $newStatus): void
    {
        switch ($newStatus) {
            case 'scheduled':
                // $this->notifyMatchParticipants($match, new MatchScheduledNotification($match));
                // $this->notifyTournamentStaff($match->tournament, new MatchScheduledAdminNotification($match));
                break;

            case 'in_progress':
                // $this->notifyMatchParticipants($match, new MatchStartedNotification($match));
                // $this->notifyTournamentStaff($match->tournament, new MatchStartedAdminNotification($match));
                
                // Notify spectators if streaming
                if ($match->is_streamed) {
                    // $this->notifyTournamentFollowers($match->tournament, new LiveMatchNotification($match));
                }
                break;

            case 'completed':
                // $this->handleMatchCompletion($match);
                break;

            case 'cancelled':
                // $this->notifyMatchParticipants($match, new MatchCancelledNotification($match));
                // $this->notifyTournamentStaff($match->tournament, new MatchCancelledAdminNotification($match));
                break;
        }
    }

    /**
     * Handle match progress status changes.
     */
    private function handleProgressStatusChange(GameMatch $match, $oldStatus, $newStatus): void
    {
        $notifications = [
            // 'warmup' => MatchWarmupStartedNotification::class,
            // 'in_progress' => MatchInProgressNotification::class,
            // 'paused' => MatchPausedNotification::class,
            // 'completed' => MatchProgressCompletedNotification::class,
        ];

        if (isset($notifications[$newStatus])) {
            $notificationClass = $notifications[$newStatus];
            // $this->notifyMatchParticipants($match, new $notificationClass($match));
        }
    }

    /**
     * Handle match completion.
     */
    private function handleMatchCompletion(GameMatch $match): void
    {
        // $this->notifyMatchParticipants($match, new MatchCompletedNotification($match));
        // $this->notifyTournamentStaff($match->tournament, new MatchCompletedAdminNotification($match));

        // Notify tournament followers about results
        // $this->notifyTournamentFollowers($match->tournament, new MatchResultsNotification($match));

        // Update player ratings if applicable
        if ($match->player1_rating_change || $match->player2_rating_change) {
            if ($match->player1) {
                // $match->player1->user->notify(new PlayerRatingChangedNotification($match->player1, $match->player1_rating_change));
            }
            if ($match->player2) {
                // $match->player2->user->notify(new PlayerRatingChangedNotification($match->player2, $match->player2_rating_change));
            }
        }
    }

    /**
     * Check if the changes include result-related fields.
     */
    private function hasResultChanges(array $changes): bool
    {
        $resultFields = ['player1_score', 'player2_score', 'game_scores', 'round_scores', 'winner_id', 'winning_team_id'];
        
        foreach ($resultFields as $field) {
            if (isset($changes[$field])) {
                return true;
            }
        }
        
        return false;
    }

    /**
     * Notify match participants (players and teams).
     */
    private function notifyMatchParticipants(GameMatch $match, $notification): void
    {
        $participants = collect();

        // Add individual players
        if ($match->player1 && $match->player1->user) {
            $participants->push($match->player1->user);
        }
        if ($match->player2 && $match->player2->user) {
            $participants->push($match->player2->user);
        }

        // Add team members
        if ($match->team1) {
            if ($match->team1->captain) {
                $participants->push($match->team1->captain);
            }
            if ($match->team1->members) {
                $participants = $participants->merge($match->team1->members);
            }
        }
        if ($match->team2) {
            if ($match->team2->captain) {
                $participants->push($match->team2->captain);
            }
            if ($match->team2->members) {
                $participants = $participants->merge($match->team2->members);
            }
        }

        // Remove duplicates and send notifications
        $participants = $participants->unique('id');
        if ($participants->isNotEmpty()) {
            Notification::send($participants, $notification);
        }
    }

    /**
     * Notify tournament staff (admins, organizers, referees).
     */
    private function notifyTournamentStaff($tournament, $notification): void
    {
        if (!$tournament) return;

        $staff = User::whereHas('roles', function ($query) {
            $query->whereIn('name', ['admin', 'super_admin', 'organizer', 'referee']);
        })->get();

        Notification::send($staff, $notification);
    }

    /**
     * Notify tournament followers/spectators.
     */
    private function notifyTournamentFollowers($tournament, $notification): void
    {
        if (!$tournament) return;

        // This could be extended to include tournament followers
        // For now, notify users who have shown interest in the tournament
        $followers = User::whereHas('players', function ($query) use ($tournament) {
            $query->where('tournament_id', $tournament->id);
        })->get();

        if ($followers->isNotEmpty()) {
            Notification::send($followers, $notification);
        }
    }
}