<?php

namespace App\Notifications\User;

use App\Models\User;
use App\Notifications\BaseNotification;
use Illuminate\Notifications\Messages\MailMessage;

class UserRegisteredNotification extends BaseNotification
{
    protected $user;
    protected $password;
    protected $username;

    public function __construct(User $user, string $password = null, string $username = null)
    {
        $this->user = $user;
        $this->password = $password;
        $this->username = $username;
    }

    protected function buildMailMessage(object $notifiable): MailMessage
    {
        $mailMessage = (new MailMessage)
            ->subject('Welcome to Tournament Management System!')
            ->greeting('Welcome, ' . $this->user->name . '!')
            ->line('Your account has been successfully created.')
            ->line('You can now participate in tournaments, join teams, and track your gaming progress.')
            ->line('Account Details:')
            ->line('• Email: ' . $this->user->email)
            ->line('• User Type: ' . ucfirst($this->user->user_type))
            ->line('• Registration Date: ' . $this->user->created_at->format('F j, Y'));

        // Add username if provided
        if ($this->username) {
            $mailMessage->line('• Username: ' . $this->username);
        }

        // Add password if provided, otherwise send password reset link
        if ($this->password) {
            $mailMessage->line('• Password: ' . $this->password)
                ->line('⚠️ Please keep your password secure and change it after your first login.');
        } else {
            $mailMessage->line('• No password was set for your account.')
                ->line('Please use the password reset link below to set your password.')
                ->action('Reset Password', url('/forgot-password'));
        }

        $mailMessage->action('Complete Your Profile', url('/profile/complete'))
            ->line('Thank you for joining our tournament community!')
            ->salutation('Best regards, Tournament Management Team');

        return $mailMessage;
    }

    public function toDatabase(object $notifiable): array
    {
        $data = [
            'user_id' => $this->user->id,
            'user_name' => $this->user->name,
            'user_type' => $this->user->user_type,
            'registration_date' => $this->user->created_at->toISOString(),
        ];

        // Add username and password to database data if provided
        if ($this->username) {
            $data['username'] = $this->username;
        }
        if ($this->password) {
            $data['password_provided'] = true;
        } else {
            $data['password_reset_required'] = true;
            $data['password_reset_url'] = url('/forgot-password');
        }

        return [
            'title' => 'Welcome to Tournament Management System!',
            'message' => 'Your account has been successfully created. Complete your profile to get started.',
            'type' => 'user_registered',
            'data' => $data,
            'action_url' => url('/profile'),
            'action_text' => 'Complete Profile',
            'priority' => 'normal',
            'category' => 'account',
        ];
    }
}