<?php

namespace App\Notifications\User;

use App\Models\Tournament;
use App\Notifications\BaseNotification;
use Illuminate\Notifications\Messages\MailMessage;

class TournamentUpdatedNotification extends BaseNotification
{
    protected Tournament $tournament;
    protected array $changes;

    public function __construct(Tournament $tournament, array $changes)
    {
        $this->tournament = $tournament;
        $this->changes = $changes;
    }

    protected function buildMailMessage(object $notifiable): MailMessage
    {
        $subject = "Tournament Updated: {$this->tournament->name}";
        
        $mailMessage = (new MailMessage)
            ->subject($subject)
            ->greeting("Hello {$notifiable->name},")
            ->line("The tournament '{$this->tournament->name}' has been updated.")
            ->line("Updated information:");

        if (isset($this->changes['start_date'])) {
            $mailMessage->line("• Start Date: {$this->tournament->start_date->format('F j, Y')}");
        }
        if (isset($this->changes['end_date'])) {
            $mailMessage->line("• End Date: {$this->tournament->end_date->format('F j, Y')}");
        }
        if (isset($this->changes['registration_end'])) {
            $mailMessage->line("• Registration Deadline: {$this->tournament->registration_end->format('F j, Y')}");
        }
        if (isset($this->changes['entry_fee'])) {
            $mailMessage->line("• Entry Fee: {$this->tournament->currency} " . number_format($this->tournament->entry_fee, 2));
        }

        $mailMessage->line("Please review the updated details.")
            ->action('View Tournament', route('tournaments.show', $this->tournament));

        return $mailMessage;
    }

    public function toArray(object $notifiable): array
    {
        return [
            'type' => 'tournament_updated',
            'title' => "Tournament Updated",
            'message' => "Tournament '{$this->tournament->name}' has been updated",
            'tournament_id' => $this->tournament->id,
            'tournament_name' => $this->tournament->name,
            'changes' => array_keys($this->changes),
            'action_url' => route('tournaments.show', $this->tournament),
            'created_at' => now()->toISOString(),
        ];
    }
}