<?php

namespace App\Notifications\User;

use App\Models\Tournament;
use App\Notifications\BaseNotification;
use Illuminate\Notifications\Messages\MailMessage;

class TournamentCreatedNotification extends BaseNotification
{
    protected Tournament $tournament;

    public function __construct(Tournament $tournament)
    {
        $this->tournament = $tournament;
    }

    protected function buildMailMessage(object $notifiable): MailMessage
    {
        $message = (new MailMessage)
            ->subject('New Tournament Available: ' . $this->tournament->name)
            ->greeting('Exciting News!')
            ->line('A new tournament has been announced and is now available for registration.')
            ->line('Tournament Details:')
            ->line('• Name: ' . $this->tournament->name)
            ->line('• Game: ' . $this->tournament->game_title)
            ->line('• Type: ' . ucfirst($this->tournament->type))
            ->line('• Format: ' . ucfirst($this->tournament->format));

        if ($this->tournament->start_date) {
            $message->line('• Start Date: ' . $this->tournament->start_date->format('F j, Y'));
        }

        if ($this->tournament->end_date) {
            $message->line('• End Date: ' . $this->tournament->end_date->format('F j, Y'));
        }

        if ($this->tournament->registration_start) {
            $message->line('• Registration Opens: ' . $this->tournament->registration_start->format('F j, Y \a\t g:i A'));
        }

        if ($this->tournament->registration_end) {
            $message->line('• Registration Closes: ' . $this->tournament->registration_end->format('F j, Y \a\t g:i A'));
        }

        if ($this->tournament->entry_fee) {
            $message->line('• Entry Fee: ' . $this->tournament->currency . ' ' . number_format($this->tournament->entry_fee, 2));
        }

        if ($this->tournament->prize_pool) {
            $message->line('• Prize Pool: ' . $this->tournament->currency . ' ' . number_format($this->tournament->prize_pool, 2));
        }

        if ($this->tournament->max_participants) {
            $message->line('• Max Participants: ' . $this->tournament->max_participants);
        }

        if ($this->tournament->location) {
            $message->line('• Location: ' . $this->tournament->location);
        }

        if ($this->tournament->is_streamed) {
            $message->line('• This tournament will be streamed live!');
        }

        $message->action('View Tournament', route('tournaments.show', $this->tournament->slug))
            ->line('Register now to secure your spot!')
            ->salutation('Best regards, Tournament Management Team');

        return $message;
    }

    public function toDatabase(object $notifiable): array
    {
        return [
            'title' => 'New Tournament Available!',
            'message' => 'The tournament "' . $this->tournament->name . '" is now accepting registrations.',
            'type' => 'tournament_created',
            'data' => [
                'tournament_id' => $this->tournament->id,
                'tournament_name' => $this->tournament->name,
                'tournament_slug' => $this->tournament->slug,
                'game_title' => $this->tournament->game_title,
                'type' => $this->tournament->type,
                'format' => $this->tournament->format,
                'status' => $this->tournament->status,
                'start_date' => $this->tournament->start_date?->toISOString(),
                'end_date' => $this->tournament->end_date?->toISOString(),
                'registration_start' => $this->tournament->registration_start?->toISOString(),
                'registration_end' => $this->tournament->registration_end?->toISOString(),
                'entry_fee' => $this->tournament->entry_fee,
                'prize_pool' => $this->tournament->prize_pool,
                'currency' => $this->tournament->currency,
                'max_participants' => $this->tournament->max_participants,
                'location' => $this->tournament->location,
                'is_streamed' => $this->tournament->is_streamed,
                'is_featured' => $this->tournament->is_featured,
                'published_at' => $this->tournament->published_at?->toISOString(),
            ],
            'action_url' => route('tournaments.show', $this->tournament->slug),
            'action_text' => 'View Tournament',
            'priority' => $this->tournament->is_featured ? 'high' : 'normal',
            'category' => 'tournament',
        ];
    }
}