<?php

namespace App\Notifications\User;

use App\Models\Payment;
use App\Notifications\BaseNotification;
use Illuminate\Notifications\Messages\MailMessage;

class TeamPaymentRefundedNotification extends BaseNotification
{
    protected Payment $payment;

    public function __construct(Payment $payment)
    {
        $this->payment = $payment;
    }

    protected function buildMailMessage(object $notifiable): MailMessage
    {
        $subject = "Team Payment Refunded: {$this->payment->reference}";
        
        $mailMessage = (new MailMessage)
            ->subject($subject)
            ->greeting("Hello {$notifiable->name},")
            ->line("Your team's payment has been refunded successfully.")
            ->line("Payment Details:")
            ->line("• Reference: {$this->payment->reference}")
            ->line("• Amount: {$this->payment->currency} " . number_format($this->payment->amount, 2))
            ->line("• Method: {$this->payment->payment_method}")
            ->line("• Team: {$this->payment->team->name}")
            ->line("• Status: Refunded");

        if ($this->payment->description) {
            $mailMessage->line("• Description: {$this->payment->description}");
        }

        if ($this->payment->refund_reason) {
            $mailMessage->line("• Refund Reason: {$this->payment->refund_reason}");
        }

        $mailMessage->line("The refund will be processed according to your payment method's timeline.")
            ->line("Thank you for your understanding.");

        return $mailMessage;
    }

    public function toArray(object $notifiable): array
    {
        return [
            'type' => 'team_payment_refunded',
            'title' => "Team Payment Refunded",
            'message' => "Your team's payment of {$this->payment->currency} " . number_format($this->payment->amount, 2) . " has been refunded",
            'payment_id' => $this->payment->id,
            'payment_reference' => $this->payment->reference,
            'payment_amount' => $this->payment->amount,
            'payment_currency' => $this->payment->currency,
            'payment_method' => $this->payment->payment_method,
            'payment_status' => $this->payment->status,
            'payment_description' => $this->payment->description,
            'refund_reason' => $this->payment->refund_reason,
            'team_id' => $this->payment->team->id,
            'team_name' => $this->payment->team->name,
            'action_url' => route('admin.payments.show', $this->payment),
            'created_at' => now()->toISOString(),
        ];
    }
} 