<?php

namespace App\Notifications\User;

use App\Models\Team;
use App\Notifications\BaseNotification;
use Illuminate\Notifications\Messages\MailMessage;

class TeamCreatedNotification extends BaseNotification
{
    protected Team $team;

    public function __construct(Team $team)
    {
        $this->team = $team;
    }

    protected function buildMailMessage(object $notifiable): MailMessage
    {
        return (new MailMessage)
            ->subject('Team Created Successfully!')
            ->greeting('Congratulations!')
            ->line('Your team has been created successfully.')
            ->line('Team Details:')
            ->line('• Team Name: ' . $this->team->name)
            ->line('• Captain: ' . $this->team->captain->name)
            ->line('• Status: ' . ucfirst($this->team->status))
            ->line('• Max Players: ' . ($this->team->max_players ?? 'Unlimited'))
            ->when($this->team->description, function ($message) {
                return $message->line('• Description: ' . $this->team->description);
            })
            ->when($this->team->payment_status, function ($message) {
                return $message->line('• Payment Status: ' . ucfirst($this->team->payment_status));
            })
            ->line('You can now invite players to join your team and register for tournaments.')
            ->action('View Team', route('admin.teams.show', $this->team->id))
            ->line('Start building your championship team!')
            ->salutation('Best regards, Tournament Management Team');
    }

    public function toDatabase(object $notifiable): array
    {
        return [
            'title' => 'Team Created Successfully!',
            'message' => 'Your team "' . $this->team->name . '" has been created and is ready for action.',
            'type' => 'team_created',
            'data' => [
                'team_id' => $this->team->id,
                'team_name' => $this->team->name,
                'captain_id' => $this->team->captain_id,
                'captain_name' => $this->team->captain->name,
                'status' => $this->team->status,
                'max_players' => $this->team->max_players,
                'description' => $this->team->description,
                'payment_status' => $this->team->payment_status,
                'created_at' => $this->team->created_at->toISOString(),
            ],
            'action_url' => route('admin.teams.show', $this->team->id),
            'action_text' => 'View Team',
            'priority' => 'high',
            'category' => 'team',
        ];
    }
}