<?php

namespace App\Notifications\User;

use App\Models\Player;
use App\Notifications\BaseNotification;
use Illuminate\Notifications\Messages\MailMessage;

class PlayerRegisteredNotification extends BaseNotification
{
    protected Player $player;

    public function __construct(Player $player)
    {
        $this->player = $player;
    }

    protected function buildMailMessage(object $notifiable): MailMessage
    {
        return (new MailMessage)
            ->subject('Player Registration Successful!')
            ->greeting('Hello ' . $this->player->user->name . '!')
            ->line('Congratulations! You have successfully registered as a player.')
            ->line('Player Details:')
            ->line('• Display Name: ' . $this->player->display_name)
            ->line('• Tournament: ' . $this->player->tournament->name)
            ->line('• Player Type: ' . ucfirst(str_replace('_', ' ', $this->player->player_type)))
            ->line('• Experience Level: ' . ucfirst($this->player->experience_level))
            ->line('• Status: ' . ucfirst($this->player->status))
            ->when($this->player->team, function ($message) {
                return $message->line('• Team: ' . $this->player->team->name);
            })
            ->action('View Player Profile', route('admin.players.show', $this->player->id))
            ->line('Good luck in the tournament!')
            ->salutation('Best regards, Tournament Management Team');
    }

    public function toDatabase(object $notifiable): array
    {
        return [
            'title' => 'Player Registration Successful!',
            'message' => 'You have successfully registered as a player for ' . $this->player->tournament->name,
            'type' => 'player_registered',
            'data' => [
                'player_id' => $this->player->id,
                'display_name' => $this->player->display_name,
                'tournament_id' => $this->player->tournament_id,
                'tournament_name' => $this->player->tournament->name,
                'player_type' => $this->player->player_type,
                'experience_level' => $this->player->experience_level,
                'status' => $this->player->status,
                'team_id' => $this->player->team_id,
                'team_name' => $this->player->team?->name,
                'registered_at' => $this->player->registered_at?->toISOString() ?? $this->player->created_at->toISOString(),
            ],
            'action_url' => route('admin.players.show', $this->player->id),
            'action_text' => 'View Profile',
            'priority' => 'high',
            'category' => 'player',
        ];
    }
}