<?php

namespace App\Notifications\User;

use App\Models\Player;
use App\Notifications\BaseNotification;
use Illuminate\Notifications\Messages\MailMessage;

class PlayerRatingUpdatedNotification extends BaseNotification
{
    public function __construct(
        protected Player $player,
        protected array $changes
    ) {}

    protected function buildMailMessage(object $notifiable): MailMessage
    {
        $ratingFields = array_filter(array_keys($this->changes), function($key) {
            return str_contains($key, 'rating') || str_contains($key, 'score');
        });

        $message = 'Your player rating has been updated.';
        if (!empty($ratingFields)) {
            $message .= ' Updated fields: ' . implode(', ', $ratingFields);
        }

        return (new MailMessage)
            ->subject('Player Rating Updated')
            ->greeting('Hello ' . ($notifiable->name ?? ''))
            ->line($message)
            ->line('Tournament: ' . ($this->player->tournament->name ?? ''))
            ->action('View Profile', url('/player/profile'))
            ->line('Your performance metrics have been updated.');
    }

    public function toDatabase(object $notifiable): array
    {
        $ratingFields = array_filter(array_keys($this->changes), function($key) {
            return str_contains($key, 'rating') || str_contains($key, 'score');
        });

        return [
            'title' => 'Player Rating Updated',
            'message' => 'Your player rating has been updated in ' . ($this->player->tournament->name ?? ''),
            'type' => 'rating_updated',
            'data' => [
                'player_id' => $this->player->id,
                'tournament_id' => $this->player->tournament->id ?? null,
                'tournament_name' => $this->player->tournament->name ?? '',
                'updated_fields' => $ratingFields,
                'changes' => $this->changes,
            ],
            'action_url' => url('/player/profile'),
            'action_text' => 'View Profile',
        ];
    }
}
