<?php

namespace App\Notifications\User;

use App\Models\Player;
use App\Notifications\BaseNotification;
use Illuminate\Notifications\Messages\MailMessage;
use Barryvdh\DomPDF\Facade\Pdf;

class PlayerDigitalIdGeneratedNotification extends BaseNotification
{
    protected $pdfContent;

    public function __construct(
        protected Player $player,
        $pdfContent = null
    ) {
        $this->pdfContent = $pdfContent;
    }

    protected function buildMailMessage(object $notifiable): MailMessage
    {
        $message = (new MailMessage)
            ->subject('Digital ID Generated')
            ->greeting('Hello ' . ($notifiable->name ?? ''))
            ->line('Your digital ID has been generated successfully!')
            ->line('Tournament: ' . ($this->player->tournament->name ?? ''))
            ->line('Your digital ID: ' . ($this->player->digital_id ?? 'N/A'))
            ->action('Download Digital ID', url('/player/digital-id'))
            ->line('You can now download your digital ID card.');

        // Attach PDF if available
        if ($this->pdfContent) {
            $filename = 'Digital_ID_' . $this->player->display_name . '_' . now()->format('Y-m-d') . '.pdf';
            $filename = preg_replace('/[^A-Za-z0-9_\-\.]/', '_', $filename);
            
            $message->attachData($this->pdfContent, $filename, [
                'mime' => 'application/pdf',
            ]);
        }

        return $message;
    }

    public function toDatabase(object $notifiable): array
    {
        return [
            'title' => 'Digital ID Generated',
            'message' => 'Your digital ID has been generated for ' . ($this->player->tournament->name ?? ''),
            'type' => 'digital_id_generated',
            'data' => [
                'player_id' => $this->player->id,
                'digital_id' => $this->player->digital_id ?? null,
                'tournament_id' => $this->player->tournament->id ?? null,
                'tournament_name' => $this->player->tournament->name ?? '',
            ],
            'action_url' => url('/player/digital-id'),
            'action_text' => 'Download Digital ID',
        ];
    }
}
