<?php

namespace App\Notifications\User;

use App\Models\Player;
use App\Notifications\BaseNotification;
use Illuminate\Notifications\Messages\MailMessage;

class PlayerCaptainStatusChangedNotification extends BaseNotification
{
    public function __construct(
        protected Player $player,
        protected bool $isCaptain
    ) {}

    protected function buildMailMessage(object $notifiable): MailMessage
    {
        $status = $this->isCaptain ? 'promoted to captain' : 'removed as captain';
        
        return (new MailMessage)
            ->subject('Captain Status Updated')
            ->greeting('Hello ' . ($notifiable->name ?? ''))
            ->line('Your captain status has been updated.')
            ->line('You have been ' . $status . ' for team "' . ($this->player->team->name ?? 'Team') . '".')
            ->line('Tournament: ' . ($this->player->tournament->name ?? ''))
            ->action('View Team Profile', url('/player/profile'))
            ->line($this->isCaptain ? 'Congratulations on becoming team captain!' : 'You are no longer the team captain.');
    }

    public function toDatabase(object $notifiable): array
    {
        $status = $this->isCaptain ? 'promoted to captain' : 'removed as captain';
        
        return [
            'title' => 'Captain Status Updated',
            'message' => 'You have been ' . $status . ' for team "' . ($this->player->team->name ?? 'Team') . '" in ' . ($this->player->tournament->name ?? ''),
            'type' => 'captain_status_changed',
            'data' => [
                'player_id' => $this->player->id,
                'team_id' => $this->player->team->id ?? null,
                'team_name' => $this->player->team->name ?? 'Team',
                'tournament_id' => $this->player->tournament->id ?? null,
                'tournament_name' => $this->player->tournament->name ?? '',
                'is_captain' => $this->isCaptain,
            ],
            'action_url' => url('/player/profile'),
            'action_text' => 'View Team Profile',
        ];
    }
}
