<?php

namespace App\Notifications\User;

use App\Models\Payment;
use App\Notifications\BaseNotification;
use Illuminate\Notifications\Messages\MailMessage;

class PaymentProcessingNotification extends BaseNotification
{
	protected Payment $payment;

	public function __construct(Payment $payment)
	{
		$this->payment = $payment;
	}

	protected function buildMailMessage(object $notifiable): MailMessage
	{
		$subject = "Payment Processing: {$this->payment->reference}";
		
		$mailMessage = (new MailMessage)
			->subject($subject)
			->greeting("Hello {$notifiable->name},")
			->line("Your payment is currently being processed.")
			->line("Payment Details:")
			->line("• Reference: {$this->payment->reference}")
			->line("• Amount: {$this->payment->currency} " . number_format($this->payment->amount, 2))
			->line("• Method: {$this->payment->payment_method}")
			->line("• Status: Processing");

		if ($this->payment->description) {
			$mailMessage->line("• Description: {$this->payment->description}");
		}

		$mailMessage->line("We will notify you once the payment is completed or if any issues arise.")
			->line("Thank you for your patience.");

		return $mailMessage;
	}

	public function toArray(object $notifiable): array
	{
		return [
			'type' => 'payment_processing',
			'title' => "Payment Processing",
			'message' => "Your payment of {$this->payment->currency} " . number_format($this->payment->amount, 2) . " is being processed",
			'payment_id' => $this->payment->id,
			'payment_reference' => $this->payment->reference,
			'payment_amount' => $this->payment->amount,
			'payment_currency' => $this->payment->currency,
			'payment_method' => $this->payment->payment_method,
			'payment_status' => $this->payment->status,
			'payment_description' => $this->payment->description,
			'action_url' => route('admin.payments.show', $this->payment),
			'created_at' => now()->toISOString(),
		];
	}
} 