<?php

namespace App\Notifications\User;

use App\Models\Payment;
use App\Notifications\BaseNotification;
use Illuminate\Notifications\Messages\MailMessage;

class PaymentInitiatedNotification extends BaseNotification
{
	protected $payment;

	public function __construct(Payment $payment)
	{
		$this->payment = $payment;
	}

	protected function buildMailMessage(object $notifiable): MailMessage
	{
		$message = (new MailMessage)
			->subject('Payment Initiated - ' . $this->payment->currency . ' ' . number_format($this->payment->amount, 2))
			->greeting('Hello ' . $this->payment->user->name . '!')
			->line('Your payment has been successfully initiated and is being processed.')
			->line('Payment Details:')
			->line('• Amount: ' . $this->payment->currency . ' ' . number_format($this->payment->amount, 2))
			->line('• Payment Method: ' . ucfirst($this->payment->payment_method))
			->line('• Reference Number: ' . $this->payment->reference_number)
			->line('• Status: ' . ucfirst($this->payment->status));

		if ($this->payment->description) {
			$message->line('• Description: ' . $this->payment->description);
		}

		if ($this->payment->team) {
			$message->line('• Team: ' . $this->payment->team->name);
		}

		$message->line('• Transaction Date: ' . $this->payment->created_at->format('F j, Y \a\t g:i A T'))
			->line('You will receive another notification once the payment is processed.')
			->action('View Payment Details', route('admin.payments.show', $this->payment->id))
			->line('If you have any questions about this payment, please contact our support team.')
			->salutation('Best regards, Tournament Management Team');

		return $message;
	}

	public function toDatabase(object $notifiable): array
	{
		return [
			'title' => 'Payment Initiated',
			'message' => 'Your payment of ' . $this->payment->currency . ' ' . number_format($this->payment->amount, 2) . ' has been initiated.',
			'type' => 'payment_initiated',
			'data' => [
				'payment_id' => $this->payment->id,
				'amount' => $this->payment->amount,
				'currency' => $this->payment->currency,
				'payment_method' => $this->payment->payment_method,
				'reference_number' => $this->payment->reference_number,
				'checkout_request_id' => $this->payment->checkout_request_id,
				'merchant_request_id' => $this->payment->merchant_request_id,
				'status' => $this->payment->status,
				'description' => $this->payment->description,
				'team_id' => $this->payment->team_id,
				'team_name' => $this->payment->team?->name,
				'user_id' => $this->payment->user_id,
				'user_name' => $this->payment->user->name,
				'user_email' => $this->payment->user->email,
				'initiated_at' => $this->payment->created_at->toISOString(),
			],
			'action_url' => route('admin.payments.show', $this->payment->id),
			'action_text' => 'View Payment',
			'priority' => 'high',
			'category' => 'payment',
		];
	}

	public function toArray(object $notifiable): array
	{
		return $this->toDatabase($notifiable);
	}
}