<?php

namespace App\Notifications\User;

use App\Models\Payment;
use App\Notifications\BaseNotification;
use Illuminate\Notifications\Messages\MailMessage;

class PaymentFailedNotification extends BaseNotification
{
	protected $payment;

	public function __construct(Payment $payment)
	{
		$this->payment = $payment;
	}

	protected function buildMailMessage(object $notifiable): MailMessage
	{
		$message = (new MailMessage)
			->subject('Payment Failed - Action Required')
			->greeting('Hello ' . $this->payment->user->name . ',')
			->line('Unfortunately, your payment could not be processed successfully.')
			->line('Payment Details:')
			->line('• Amount: ' . $this->payment->currency . ' ' . number_format($this->payment->amount, 2))
			->line('• Payment Method: ' . ucfirst($this->payment->payment_method))
			->line('• Reference Number: ' . $this->payment->reference_number)
			->line('• Attempted Date: ' . $this->payment->created_at->format('F j, Y \a\t g:i A T'));

		if ($this->payment->failure_reason) {
			$message->line('• Failure Reason: ' . $this->payment->failure_reason);
		}

		if ($this->payment->description) {
			$message->line('• Purpose: ' . $this->payment->description);
		}

		if ($this->payment->team) {
			$message->line('• Team: ' . $this->payment->team->name);
		}

		$message->line('What happens next?')
			->line('• Your account has not been charged')
			->line('• You can retry the payment with the same or different payment method')
			->line('• If you continue to experience issues, please contact our support team')
			->line('• Your registration/booking is still held for 24 hours');

		$commonReasons = $this->getCommonFailureReasons();
		if (!empty($commonReasons)) {
			$message->line('Common reasons for payment failure:');
			foreach ($commonReasons as $reason) {
				$message->line('• ' . $reason);
			}
		}

		$message->action('Retry Payment', route('payments.retry', $this->payment->id))
			->line('If you need assistance, please contact our support team with reference number: ' . $this->payment->reference_number)
			->salutation('Best regards, Tournament Management Team');

		return $message;
	}

	public function toDatabase(object $notifiable): array
	{
		return [
			'title' => 'Payment Failed',
			'message' => 'Your payment of ' . $this->payment->currency . ' ' . number_format($this->payment->amount, 2) . ' could not be processed.',
			'type' => 'payment_failed',
			'data' => [
				'payment_id' => $this->payment->id,
				'amount' => $this->payment->amount,
				'currency' => $this->payment->currency,
				'payment_method' => $this->payment->payment_method,
				'reference_number' => $this->payment->reference_number,
				'checkout_request_id' => $this->payment->checkout_request_id,
				'merchant_request_id' => $this->payment->merchant_request_id,
				'status' => $this->payment->status,
				'failure_reason' => $this->payment->failure_reason,
				'description' => $this->payment->description,
				'team_id' => $this->payment->team_id,
				'team_name' => $this->payment->team?->name,
				'user_id' => $this->payment->user_id,
				'user_name' => $this->payment->user->name,
				'user_email' => $this->payment->user->email,
				'failed_at' => $this->payment->updated_at->toISOString(),
				'retry_available' => true,
				'support_contact' => true,
			],
			'action_url' => route('payments.retry', $this->payment->id),
			'action_text' => 'Retry Payment',
			'priority' => 'high',
			'category' => 'payment',
		];
	}

	private function getCommonFailureReasons(): array
	{
		return [
			'Insufficient funds in your account',
			'Incorrect card details or expired card',
			'Payment method not supported',
			'Bank declined the transaction',
			'Network connectivity issues',
			'Security restrictions from your bank',
		];
	}

	public function toArray(object $notifiable): array
	{
		return $this->toDatabase($notifiable);
	}
}