<?php

namespace App\Notifications\User;

use App\Models\Payment;
use App\Notifications\BaseNotification;
use Illuminate\Notifications\Messages\MailMessage;

class PaymentCompletedNotification extends BaseNotification
{
	protected $payment;

	public function __construct(Payment $payment)
	{
		$this->payment = $payment;
	}

	protected function buildMailMessage(object $notifiable): MailMessage
	{
		$message = (new MailMessage)
			->subject('Payment Successful - ' . $this->payment->currency . ' ' . number_format($this->payment->amount, 2))
			->greeting('Great news, ' . $this->payment->user->name . '!')
			->line('Your payment has been successfully processed and completed.')
			->line('Payment Summary:')
			->line('• Amount Paid: ' . $this->payment->currency . ' ' . number_format($this->payment->amount, 2))
			->line('• Payment Method: ' . ucfirst($this->payment->payment_method))
			->line('• Reference Number: ' . $this->payment->reference_number)
			->line('• Transaction ID: ' . ($this->payment->checkout_request_id ?? $this->payment->merchant_request_id))
			->line('• Payment Date: ' . $this->payment->paid_at->format('F j, Y \a\t g:i A T'));

		if ($this->payment->description) {
			$message->line('• Purpose: ' . $this->payment->description);
		}

		if ($this->payment->team) {
			$message->line('• Team: ' . $this->payment->team->name)
				->line('Your team registration is now complete and active.');
		}

		$message->line('Receipt Details:')
			->line('A detailed receipt has been generated for your records.')
			->action('Download Receipt', route('payments.receipt', $this->payment->id))
			->line('Thank you for your payment! You can now access all premium features.')
			->line('If you need any assistance, our support team is here to help.')
			->salutation('Best regards, Tournament Management Team');

		return $message;
	}

	public function toDatabase(object $notifiable): array
	{
		return [
			'title' => 'Payment Completed Successfully!',
			'message' => 'Your payment of ' . $this->payment->currency . ' ' . number_format($this->payment->amount, 2) . ' has been completed.',
			'type' => 'payment_completed',
			'data' => [
				'payment_id' => $this->payment->id,
				'amount' => $this->payment->amount,
				'currency' => $this->payment->currency,
				'payment_method' => $this->payment->payment_method,
				'reference_number' => $this->payment->reference_number,
				'checkout_request_id' => $this->payment->checkout_request_id,
				'merchant_request_id' => $this->payment->merchant_request_id,
				'status' => $this->payment->status,
				'description' => $this->payment->description,
				'team_id' => $this->payment->team_id,
				'team_name' => $this->payment->team?->name,
				'user_id' => $this->payment->user_id,
				'user_name' => $this->payment->user->name,
				'user_email' => $this->payment->user->email,
				'paid_at' => $this->payment->paid_at?->toISOString(),
				'completed_at' => now()->toISOString(),
				'receipt_available' => true,
			],
			'action_url' => route('payments.receipt', $this->payment->id),
			'action_text' => 'Download Receipt',
			'priority' => 'high',
			'category' => 'payment',
		];
	}

	public function toArray(object $notifiable): array
	{
		return $this->toDatabase($notifiable);
	}
}