<?php

namespace App\Notifications\User;

use App\Models\GameMatch;
use App\Notifications\BaseNotification;
use Illuminate\Notifications\Messages\MailMessage;

class MatchVerificationUpdatedNotification extends BaseNotification
{
    protected GameMatch $match;
    protected string $verificationStatus;

    public function __construct(GameMatch $match, string $verificationStatus)
    {
        $this->match = $match;
        $this->verificationStatus = $verificationStatus;
    }

    protected function buildMailMessage(object $notifiable): MailMessage
    {
        $subject = "Match Verification Updated: {$this->match->team1->name} vs {$this->match->team2->name}";
        
        return (new MailMessage)
            ->subject($subject)
            ->greeting("Hello {$notifiable->name},")
            ->line("The verification status for a match you're involved in has been updated.")
            ->line("Match Details:")
            ->line("• Teams: {$this->match->team1->name} vs {$this->match->team2->name}")
            ->line("• Tournament: {$this->match->tournament->name}")
            ->line("• Verification Status: {$this->verificationStatus}")
            ->action('View Match Details', route('matches.show', $this->match));
    }

    public function toArray(object $notifiable): array
    {
        return [
            'type' => 'match_verification_updated',
            'title' => "Match Verification Updated",
            'message' => "Verification status for {$this->match->team1->name} vs {$this->match->team2->name} updated to {$this->verificationStatus}",
            'match_id' => $this->match->id,
            'team1_name' => $this->match->team1->name,
            'team2_name' => $this->match->team2->name,
            'tournament_name' => $this->match->tournament->name,
            'verification_status' => $this->verificationStatus,
            'action_url' => route('matches.show', $this->match),
            'created_at' => now()->toISOString(),
        ];
    }
}