<?php

namespace App\Notifications\User;

use App\Models\GameMatch;
use App\Notifications\BaseNotification;
use Illuminate\Notifications\Messages\MailMessage;

class MatchCompletedNotification extends BaseNotification
{
    protected GameMatch $match;

    public function __construct(GameMatch $match)
    {
        $this->match = $match;
    }

    protected function buildMailMessage(object $notifiable): MailMessage
    {
        $subject = "Match Completed: {$this->match->team1->name} vs {$this->match->team2->name}";
        
        $mailMessage = (new MailMessage)
            ->subject($subject)
            ->greeting("Hello {$notifiable->name},")
            ->line("A match you're involved in has been completed.")
            ->line("Match Details:")
            ->line("• Teams: {$this->match->team1->name} vs {$this->match->team2->name}")
            ->line("• Tournament: {$this->match->tournament->name}");

        if ($this->match->team1_score !== null && $this->match->team2_score !== null) {
            $mailMessage->line("• Final Score: {$this->match->team1_score} - {$this->match->team2_score}");
        }

        if ($this->match->winner_team_id) {
            $winner = $this->match->winner_team_id === $this->match->team1_id 
                ? $this->match->team1->name 
                : $this->match->team2->name;
            $mailMessage->line("• Winner: {$winner}");
        }

        $mailMessage->action('View Match Details', route('matches.show', $this->match));

        return $mailMessage;
    }

    public function toArray(object $notifiable): array
    {
        return [
            'type' => 'match_completed',
            'title' => "Match Completed",
            'message' => "Match between {$this->match->team1->name} and {$this->match->team2->name} has been completed",
            'match_id' => $this->match->id,
            'team1_name' => $this->match->team1->name,
            'team2_name' => $this->match->team2->name,
            'team1_score' => $this->match->team1_score,
            'team2_score' => $this->match->team2_score,
            'winner_team_id' => $this->match->winner_team_id,
            'tournament_name' => $this->match->tournament->name,
            'action_url' => route('matches.show', $this->match),
            'created_at' => now()->toISOString(),
        ];
    }
}