<?php

namespace App\Notifications\User;

use App\Models\GameMatch;
use App\Notifications\BaseNotification;
use Illuminate\Notifications\Messages\MailMessage;

class MatchCancelledNotification extends BaseNotification
{
    protected GameMatch $match;
    protected ?string $reason;

    public function __construct(GameMatch $match, ?string $reason = null)
    {
        $this->match = $match;
        $this->reason = $reason;
    }

    protected function buildMailMessage(object $notifiable): MailMessage
    {
        $subject = "Match Cancelled: {$this->match->team1->name} vs {$this->match->team2->name}";
        
        $mailMessage = (new MailMessage)
            ->subject($subject)
            ->greeting("Hello {$notifiable->name},")
            ->line("A match you're involved in has been cancelled.")
            ->line("Match Details:")
            ->line("• Teams: {$this->match->team1->name} vs {$this->match->team2->name}")
            ->line("• Tournament: {$this->match->tournament->name}")
            ->line("• Originally scheduled: {$this->match->start_time->format('F j, Y g:i A')}");

        if ($this->reason) {
            $mailMessage->line("• Cancellation reason: {$this->reason}");
        }

        $mailMessage->line("We apologize for any inconvenience caused.")
            ->action('View Tournament', route('tournaments.show', $this->match->tournament));

        return $mailMessage;
    }

    public function toArray(object $notifiable): array
    {
        return [
            'type' => 'match_cancelled',
            'title' => "Match Cancelled",
            'message' => "Match between {$this->match->team1->name} and {$this->match->team2->name} has been cancelled",
            'match_id' => $this->match->id,
            'team1_name' => $this->match->team1->name,
            'team2_name' => $this->match->team2->name,
            'tournament_name' => $this->match->tournament->name,
            'original_start_time' => $this->match->start_time->toISOString(),
            'cancellation_reason' => $this->reason,
            'action_url' => route('tournaments.show', $this->match->tournament),
            'created_at' => now()->toISOString(),
        ];
    }
}