<?php

namespace App\Notifications\User;

use App\Notifications\BaseNotification;
use Illuminate\Notifications\Messages\MailMessage;

class EmailAlreadyTakenNotification extends BaseNotification
{
    protected $attemptedRegistrationData;
    protected $tournamentName;

    public function __construct(array $attemptedRegistrationData, string $tournamentName)
    {
        $this->attemptedRegistrationData = $attemptedRegistrationData;
        $this->tournamentName = $tournamentName;
    }

    protected function buildMailMessage(object $notifiable): MailMessage
    {
        return (new MailMessage)
            ->subject('Someone Tried to Register with Your Email')
            ->greeting('Hello ' . $notifiable->name . '!')
            ->line('We noticed that someone attempted to register for a tournament using your email address.')
            ->line('Tournament: ' . $this->tournamentName)
            ->line('Registration Details:')
            ->line('• Name: ' . ($this->attemptedRegistrationData['name'] ?? 'Not provided'))
            ->line('• Phone: ' . ($this->attemptedRegistrationData['phone'] ?? 'Not provided'))
            ->line('• Team: ' . ($this->attemptedRegistrationData['team_name'] ?? 'Not provided'))
            ->line('• Attempted At: ' . now()->format('F j, Y g:i A'))
            ->line('If this was you, please log in to your existing account instead of creating a new one.')
            ->line('If this was not you, please contact our support team immediately.')
            ->action('Login to Your Account', url('/login'))
            ->line('Thank you for your attention to this matter.')
            ->salutation('Best regards, Tournament Management Team');
    }

    public function toDatabase(object $notifiable): array
    {
        return [
            'title' => 'Email Registration Attempt',
            'message' => 'Someone tried to register for ' . $this->tournamentName . ' using your email address.',
            'type' => 'email_already_taken',
            'data' => [
                'tournament_name' => $this->tournamentName,
                'attempted_registration' => $this->attemptedRegistrationData,
                'attempted_at' => now()->toISOString(),
            ],
            'action_url' => url('/login'),
            'action_text' => 'Login to Account',
            'priority' => 'high',
            'category' => 'security',
        ];
    }
}
