<?php

namespace App\Notifications;

use Illuminate\Bus\Queueable;
use Illuminate\Contracts\Queue\ShouldQueue;
use Illuminate\Notifications\Messages\MailMessage;
use Illuminate\Notifications\Notification;
use Illuminate\Support\Facades\Log;
use App\Models\NotificationSetting;
use App\Models\UserNotificationPreference;
use Illuminate\Support\Str;

abstract class BaseNotification extends Notification implements ShouldQueue
{
    use Queueable;

    /**
     * The number of times the notification may be attempted.
     *
     * @var int
     */
    public $tries = 3;

    /**
     * The number of seconds the notification can run before timing out.
     *
     * @var int
     */
    public $timeout = 120;

    /**
     * The number of seconds to wait before retrying the notification.
     *
     * @var array<int>
     */
    public $backoff = [60, 300];

    /**
     * Determine which queue this notification should be sent to based on type.
     *
     * @return string
     */
    public function viaQueues(): array
    {
        $queue = $this->determineQueue();
        
        return [
            'mail' => $queue,
            'database' => $queue,
        ];
    }

    /**
     * Determine which queue this notification should be sent to based on type.
     */
    protected function determineQueue(): string
    {
        $className = class_basename($this);

        // High priority queue for critical notifications
        if ($this->isHighPriority($className)) {
            return 'high';
        }

        // Low priority queue for general updates
        if ($this->isLowPriority($className)) {
            return 'low';
        }

        // Default queue for everything else
        return 'default';
    }

    /**
     * Check if notification is high priority.
     */
    protected function isHighPriority(string $className): bool
    {
        $highPriorityKeywords = [
            'Payment',
            'Verification',
            'Otp',
            'Security',
            'Critical',
            'Urgent',
        ];

        foreach ($highPriorityKeywords as $keyword) {
            if (str_contains($className, $keyword)) {
                return true;
            }
        }

        return false;
    }

    /**
     * Check if notification is low priority.
     */
    protected function isLowPriority(string $className): bool
    {
        $lowPriorityKeywords = [
            'ProfileUpdated',
            'PasswordChanged',
            'RoleChanged',
            'Updated',
        ];

        foreach ($lowPriorityKeywords as $keyword) {
            if (str_contains($className, $keyword)) {
                return true;
            }
        }

        return false;
    }

    /**
     * Get the notification's delivery channels.
     */
    public function via(object $notifiable): array
    {
        // Check if this notification can be sent
        if (!$this->canSend($notifiable)) {
            return [];
        }

        // Only send mail if the notifiable has a valid email address
        if ($this->hasValidEmail($notifiable)) {
            return ['mail', 'database'];
        }
        
        // If no valid email, only send to database
        return ['database'];
    }

    /**
     * Check if this notification can be sent to the notifiable.
     * Checks both system-wide settings and user preferences.
     */
    protected function canSend(object $notifiable): bool
    {
        $notificationType = $this->getNotificationType();
        $category = $this->getNotificationCategory();

        // Check system-wide setting first
        if (!NotificationSetting::isNotificationEnabled($notificationType)) {
            Log::info('Notification blocked by system setting', [
                'notification_type' => $notificationType,
                'user_id' => $notifiable->id ?? 'unknown',
            ]);
            return false;
        }

        // Check user preference if notifiable has an ID (is a User)
        if (isset($notifiable->id) && method_exists($notifiable, 'notificationPreferences')) {
            if (!UserNotificationPreference::isEnabledForUser($notifiable->id, $notificationType, $category)) {
                Log::info('Notification blocked by user preference', [
                    'notification_type' => $notificationType,
                    'category' => $category,
                    'user_id' => $notifiable->id,
                ]);
                return false;
            }
        }

        return true;
    }

    /**
     * Get the notification type identifier.
     */
    protected function getNotificationType(): string
    {
        $className = class_basename($this);
        return Str::snake($className);
    }

    /**
     * Get the notification category based on class namespace or name.
     */
    protected function getNotificationCategory(): string
    {
        $className = class_basename($this);
        $namespace = (new \ReflectionClass($this))->getNamespaceName();

        // Check if it's in Admin namespace
        if (str_contains($namespace, 'Admin')) {
            return 'admin';
        }

        // Determine category based on class name keywords
        $categoryKeywords = [
            'Team' => 'team',
            'Player' => 'player',
            'Payment' => 'payment',
            'Tournament' => 'tournament',
            'Match' => 'match',
            'Official' => 'official',
            'User' => 'user',
            'Certificate' => 'certificate',
            'Invitation' => 'invitation',
            'Otp' => 'auth',
            'Password' => 'auth',
            'Verification' => 'auth',
            'Email' => 'auth',
        ];

        foreach ($categoryKeywords as $keyword => $category) {
            if (str_contains($className, $keyword)) {
                return $category;
            }
        }

        return 'system';
    }

    /**
     * Check if the notifiable has a valid email address.
     */
    protected function hasValidEmail(object $notifiable): bool
    {
        return !empty($notifiable->email) && filter_var($notifiable->email, FILTER_VALIDATE_EMAIL);
    }

    /**
     * Get the mail representation of the notification.
     * This method includes error handling to suppress mail failures.
     */
    public function toMail(object $notifiable): MailMessage
    {
        try {
            // Double-check email validity before attempting to send
            if (!$this->hasValidEmail($notifiable)) {
                Log::warning('Attempted to send email to user with invalid email', [
                    'notification_class' => static::class,
                    'user_id' => $notifiable->id ?? 'unknown',
                    'user_email' => $notifiable->email ?? 'null'
                ]);
                return $this->getFallbackMailMessage();
            }
            
            return $this->buildMailMessage($notifiable);
        } catch (\Exception $e) {
            $this->logMailError($e, $notifiable);
            return $this->getFallbackMailMessage();
        }
    }

    /**
     * Build the mail message. Override this method in child classes.
     */
    abstract protected function buildMailMessage(object $notifiable): MailMessage;

    /**
     * Get a fallback mail message when the main one fails.
     */
    protected function getFallbackMailMessage(): MailMessage
    {
        return (new MailMessage)
            ->subject('Notification')
            ->line('You have a new notification.');
    }

    /**
     * Log mail errors for debugging.
     */
    protected function logMailError(\Exception $e, object $notifiable): void
    {
        Log::warning('Failed to send notification email', [
            'notification_class' => static::class,
            'user_id' => $notifiable->id ?? 'unknown',
            'user_email' => $notifiable->email ?? 'unknown',
            'error' => $e->getMessage(),
            'error_code' => $e->getCode(),
            'file' => $e->getFile(),
            'line' => $e->getLine()
        ]);
    }

    /**
     * Get the array representation of the notification.
     */
    public function toArray(object $notifiable): array
    {
        return [];
    }
}
