<?php

namespace App\Notifications\Admin;

use App\Models\User;
use App\Notifications\BaseNotification;
use Illuminate\Notifications\Messages\MailMessage;

class UserProfileUpdatedAdminNotification extends BaseNotification
{
	protected User $user;
	protected array $changes;

	public function __construct(User $user, array $changes)
	{
		$this->user = $user;
		$this->changes = $changes;
	}

	protected function buildMailMessage(object $notifiable): MailMessage
	{
		$subject = "User Profile Updated: {$this->user->name}";
		
		$mailMessage = (new MailMessage)
			->subject($subject)
			->greeting("Hello {$notifiable->name},")
			->line("A user has updated their profile.")
			->line("User: {$this->user->name} ({$this->user->email})")
			->line("Updated fields:");

		foreach ($this->changes as $field => $value) {
			if (!in_array($field, ['password', 'remember_token', 'email_verified_at'])) {
				$mailMessage->line("• " . ucfirst(str_replace('_', ' ', $field)));
			}
		}

		$mailMessage->action('View User Profile', route('admin.users.show', $this->user));

		return $mailMessage;
	}

	public function toDatabase(object $notifiable): array
	{
		return [
			'type' => 'user_profile_updated_admin',
			'title' => "User Profile Updated",
			'message' => "User {$this->user->name} has updated their profile",
			'user_id' => $this->user->id,
			'user_name' => $this->user->name,
			'user_email' => $this->user->email,
			'changes' => array_keys($this->changes),
			'action_url' => route('admin.users.show', $this->user),
			'created_at' => now()->toISOString(),
		];
	}
}