<?php

namespace App\Notifications\Admin;

use App\Models\Tournament;
use App\Notifications\BaseNotification;
use Illuminate\Notifications\Messages\MailMessage;

class TournamentUpdatedAdminNotification extends BaseNotification
{
    protected Tournament $tournament;
    protected array $changes;

    public function __construct(Tournament $tournament, array $changes = [])
    {
        $this->tournament = $tournament;
        $this->changes = $changes;
    }

    protected function buildMailMessage(object $notifiable): MailMessage
    {
        $subject = "Tournament Updated: {$this->tournament->name}";
        
        $mailMessage = (new MailMessage)
            ->subject($subject)
            ->greeting('Hello ' . ($notifiable->name ?? ''))
            ->line('A tournament has been updated on the platform.')
            ->line('Tournament Details:')
            ->line('• Name: ' . $this->tournament->name)
            ->line('• Status: ' . ucfirst($this->tournament->status))
            ->line('• Updated At: ' . $this->tournament->updated_at->format('F j, Y g:i A'));

        if (!empty($this->changes)) {
            $mailMessage->line('Changes Made:');
            foreach ($this->changes as $field => $value) {
                $mailMessage->line('• ' . ucfirst(str_replace('_', ' ', $field)) . ': ' . $value);
            }
        }

        $mailMessage->action('View Tournament', url('/admin/tournaments/' . $this->tournament->id))
            ->line('Please review the changes and take any necessary actions.');

        return $mailMessage;
    }

    public function toDatabase(object $notifiable): array
    {
        return [
            'title' => 'Tournament Updated',
            'message' => 'Tournament "' . $this->tournament->name . '" has been updated',
            'type' => 'tournament_updated_admin',
            'data' => [
                'tournament_id' => $this->tournament->id,
                'tournament_name' => $this->tournament->name,
                'status' => $this->tournament->status,
                'changes' => $this->changes,
                'updated_at' => $this->tournament->updated_at->toISOString(),
            ],
            'action_url' => url('/admin/tournaments/' . $this->tournament->id),
            'action_text' => 'View Tournament',
        ];
    }
} 