<?php

namespace App\Notifications\Admin;

use App\Models\Tournament;
use App\Notifications\BaseNotification;
use Illuminate\Notifications\Messages\MailMessage;

class TournamentStatusChangedAdminNotification extends BaseNotification
{
    protected Tournament $tournament;
    protected string $oldStatus;
    protected string $newStatus;

    public function __construct(Tournament $tournament, string $oldStatus, string $newStatus)
    {
        $this->tournament = $tournament;
        $this->oldStatus = $oldStatus;
        $this->newStatus = $newStatus;
    }

    protected function buildMailMessage(object $notifiable): MailMessage
    {
        $subject = "Tournament Status Changed: {$this->tournament->name}";
        
        return (new MailMessage)
            ->subject($subject)
            ->greeting('Hello ' . ($notifiable->name ?? ''))
            ->line('A tournament status has been changed on the platform.')
            ->line('Tournament Details:')
            ->line('• Name: ' . $this->tournament->name)
            ->line('• Old Status: ' . ucfirst($this->oldStatus))
            ->line('• New Status: ' . ucfirst($this->newStatus))
            ->line('• Changed At: ' . $this->tournament->updated_at->format('F j, Y g:i A'))
            ->action('View Tournament', url('/admin/tournaments/' . $this->tournament->id))
            ->line('Please review the status change and take any necessary actions.');
    }

    public function toDatabase(object $notifiable): array
    {
        return [
            'title' => 'Tournament Status Changed',
            'message' => 'Tournament "' . $this->tournament->name . '" status changed from ' . ucfirst($this->oldStatus) . ' to ' . ucfirst($this->newStatus),
            'type' => 'tournament_status_changed_admin',
            'data' => [
                'tournament_id' => $this->tournament->id,
                'tournament_name' => $this->tournament->name,
                'old_status' => $this->oldStatus,
                'new_status' => $this->newStatus,
                'changed_at' => $this->tournament->updated_at->toISOString(),
            ],
            'action_url' => url('/admin/tournaments/' . $this->tournament->id),
            'action_text' => 'View Tournament',
        ];
    }
} 