<?php

namespace App\Notifications\Admin;

use App\Models\Tournament;
use App\Notifications\BaseNotification;
use Illuminate\Notifications\Messages\MailMessage;

class TournamentCreatedAdminNotification extends BaseNotification
{
    protected Tournament $tournament;

    public function __construct(Tournament $tournament)
    {
        $this->tournament = $tournament;
    }

    protected function buildMailMessage(object $notifiable): MailMessage
    {
        $subject = "New Tournament Created: {$this->tournament->name}";
        
        return (new MailMessage)
            ->subject($subject)
            ->greeting('Hello ' . ($notifiable->name ?? ''))
            ->line('A new tournament has been created on the platform.')
            ->line('Tournament Details:')
            ->line('• Name: ' . $this->tournament->name)
            ->line('• Description: ' . ($this->tournament->description ?? 'No description'))
            ->line('• Start Date: ' . ($this->tournament->start_date ? $this->tournament->start_date->format('F j, Y') : 'Not set'))
            ->line('• End Date: ' . ($this->tournament->end_date ? $this->tournament->end_date->format('F j, Y') : 'Not set'))
            ->line('• Entry Fee: ' . ($this->tournament->entry_fee ? $this->tournament->entry_fee . ' ' . ($this->tournament->currency ?? 'KES') : 'Free'))
            ->line('• Status: ' . ucfirst($this->tournament->status))
            ->line('• Created At: ' . $this->tournament->created_at->format('F j, Y g:i A'))
            ->action('View Tournament', url('/admin/tournaments/' . $this->tournament->id))
            ->line('Please review the tournament details and approve if necessary.');
    }

    public function toDatabase(object $notifiable): array
    {
        return [
            'title' => 'New Tournament Created',
            'message' => 'New tournament "' . $this->tournament->name . '" has been created',
            'type' => 'tournament_created_admin',
            'data' => [
                'tournament_id' => $this->tournament->id,
                'tournament_name' => $this->tournament->name,
                'description' => $this->tournament->description,
                'start_date' => $this->tournament->start_date?->toISOString(),
                'end_date' => $this->tournament->end_date?->toISOString(),
                'entry_fee' => $this->tournament->entry_fee,
                'currency' => $this->tournament->currency,
                'status' => $this->tournament->status,
                'created_at' => $this->tournament->created_at->toISOString(),
            ],
            'action_url' => url('/admin/tournaments/' . $this->tournament->id),
            'action_text' => 'View Tournament',
        ];
    }
} 