<?php

namespace App\Notifications\Admin;

use App\Models\Player;
use App\Notifications\BaseNotification;
use Illuminate\Notifications\Messages\MailMessage;

class PlayerStatusChangedAdminNotification extends BaseNotification
{
    public function __construct(
        protected Player $player,
        protected string $oldStatus,
        protected string $newStatus
    ) {}

    protected function buildMailMessage(object $notifiable): MailMessage
    {
        return (new MailMessage)
            ->subject('Player Status Changed')
            ->greeting('Hello ' . ($notifiable->name ?? ''))
            ->line('A player status has been changed.')
            ->line('Player: ' . ($this->player->display_name ?? $this->player->user->name ?? 'Unknown'))
            ->line('Old Status: ' . $this->oldStatus)
            ->line('New Status: ' . $this->newStatus)
            ->line('Tournament: ' . ($this->player->tournament->name ?? ''))
            ->action('View Player', url('/admin/players/' . $this->player->id))
            ->line('Please review this change.');
    }

    public function toDatabase(object $notifiable): array
    {
        return [
            'title' => 'Player Status Changed',
            'message' => 'Player ' . ($this->player->display_name ?? $this->player->user->name ?? 'Unknown') . ' status changed from ' . $this->oldStatus . ' to ' . $this->newStatus . ' in ' . ($this->player->tournament->name ?? ''),
            'type' => 'player_status_changed',
            'data' => [
                'player_id' => $this->player->id,
                'player_name' => $this->player->display_name ?? $this->player->user->name ?? 'Unknown',
                'old_status' => $this->oldStatus,
                'new_status' => $this->newStatus,
                'tournament_id' => $this->player->tournament->id ?? null,
                'tournament_name' => $this->player->tournament->name ?? '',
            ],
            'action_url' => url('/admin/players/' . $this->player->id),
            'action_text' => 'View Player',
        ];
    }
}
