<?php

namespace App\Notifications\Admin;

use App\Models\Player;
use App\Notifications\BaseNotification;
use Illuminate\Notifications\Messages\MailMessage;

class PlayerRegisteredAdminNotification extends BaseNotification
{
    protected Player $player;

    public function __construct(Player $player)
    {
        $this->player = $player;
    }

    protected function buildMailMessage(object $notifiable): MailMessage
    {
        $subject = "New Player Registration: {$this->player->display_name}";
        
        return (new MailMessage)
            ->subject($subject)
            ->greeting('Hello ' . ($notifiable->name ?? ''))
            ->line('A new player has registered on the platform.')
            ->line('Player Details:')
            ->line('• Name: ' . $this->player->display_name)
            ->line('• User: ' . ($this->player->user->name ?? 'Unknown'))
            ->line('• User Email: ' . ($this->player->user->email ?? 'Unknown'))
            ->line('• Team: ' . ($this->player->team->name ?? 'Unknown'))
            ->line('• Tournament: ' . ($this->player->tournament->name ?? 'Unknown'))
            ->line('• Position: ' . ($this->player->football_position ?? 'Unknown'))
            ->line('• Registration Date: ' . $this->player->created_at->format('F j, Y g:i A'))
            ->action('View Player', url('/admin/players/' . $this->player->id))
            ->line('Please review the player registration and verify their details.');
    }

    public function toDatabase(object $notifiable): array
    {
        return [
            'title' => 'New Player Registration',
            'message' => 'New player ' . $this->player->display_name . ' registered for team ' . ($this->player->team->name ?? 'Unknown'),
            'type' => 'player_registered_admin',
            'data' => [
                'player_id' => $this->player->id,
                'player_name' => $this->player->display_name,
                'user_id' => $this->player->user->id ?? null,
                'user_name' => $this->player->user->name ?? 'Unknown',
                'user_email' => $this->player->user->email ?? 'Unknown',
                'team_id' => $this->player->team->id ?? null,
                'team_name' => $this->player->team->name ?? 'Unknown',
                'tournament_id' => $this->player->tournament->id ?? null,
                'tournament_name' => $this->player->tournament->name ?? 'Unknown',
                'position' => $this->player->football_position,
                'registration_date' => $this->player->created_at->toISOString(),
            ],
            'action_url' => url('/admin/players/' . $this->player->id),
            'action_text' => 'View Player',
        ];
    }
}