<?php

namespace App\Notifications\Admin;

use App\Models\Player;
use App\Notifications\BaseNotification;
use Illuminate\Notifications\Messages\MailMessage;
use Barryvdh\DomPDF\Facade\Pdf;

class PlayerDigitalIdGeneratedAdminNotification extends BaseNotification
{
    protected Player $player;
    protected $pdfContent;

    public function __construct(Player $player, $pdfContent = null)
    {
        $this->player = $player;
        $this->pdfContent = $pdfContent;
    }

    protected function buildMailMessage(object $notifiable): MailMessage
    {
        $subject = "Digital ID Generated: {$this->player->display_name}";
        
        $message = (new MailMessage)
            ->subject($subject)
            ->greeting('Hello ' . ($notifiable->name ?? ''))
            ->line('A player\'s digital ID has been generated successfully.')
            ->line('Player Details:')
            ->line('• Name: ' . $this->player->display_name)
            ->line('• User: ' . ($this->player->user->name ?? 'Unknown'))
            ->line('• User Email: ' . ($this->player->user->email ?? 'Unknown'))
            ->line('• Team: ' . ($this->player->team->name ?? 'Unknown'))
            ->line('• Tournament: ' . ($this->player->tournament->name ?? 'Unknown'))
            ->line('• Digital ID: ' . ($this->player->digital_id ?? 'N/A'))
            ->line('• Generation Date: ' . now()->format('F j, Y g:i A'))
            ->action('View Player', url('/admin/players/' . $this->player->id))
            ->line('The player has been notified and can now download their digital ID card.');

        // Attach PDF if available
        if ($this->pdfContent) {
            $filename = 'Digital_ID_' . $this->player->display_name . '_' . now()->format('Y-m-d') . '.pdf';
            $filename = preg_replace('/[^A-Za-z0-9_\-\.]/', '_', $filename);
            
            $message->attachData($this->pdfContent, $filename, [
                'mime' => 'application/pdf',
            ]);
        }

        return $message;
    }

    public function toDatabase(object $notifiable): array
    {
        return [
            'title' => 'Digital ID Generated',
            'message' => 'Digital ID generated for player ' . $this->player->display_name . ' in team ' . ($this->player->team->name ?? 'Unknown'),
            'type' => 'digital_id_generated_admin',
            'data' => [
                'player_id' => $this->player->id,
                'player_name' => $this->player->display_name,
                'user_id' => $this->player->user->id ?? null,
                'user_name' => $this->player->user->name ?? 'Unknown',
                'user_email' => $this->player->user->email ?? 'Unknown',
                'team_id' => $this->player->team->id ?? null,
                'team_name' => $this->player->team->name ?? 'Unknown',
                'tournament_id' => $this->player->tournament->id ?? null,
                'tournament_name' => $this->player->tournament->name ?? 'Unknown',
                'digital_id' => $this->player->digital_id ?? null,
                'generation_date' => now()->toISOString(),
            ],
            'action_url' => url('/admin/players/' . $this->player->id),
            'action_text' => 'View Player',
        ];
    }
}
