<?php

namespace App\Notifications\Admin;

use App\Models\Payment;
use App\Notifications\BaseNotification;
use Illuminate\Notifications\Messages\MailMessage;

class PaymentProcessingAdminNotification extends BaseNotification
{
    protected Payment $payment;

    public function __construct(Payment $payment)
    {
        $this->payment = $payment;
    }

    protected function buildMailMessage(object $notifiable): MailMessage
    {
        $subject = "Payment Processing: {$this->payment->amount} {$this->payment->currency}";
        
        return (new MailMessage)
            ->subject($subject)
            ->greeting('Hello ' . ($notifiable->name ?? ''))
            ->line('A payment is now being processed on the platform.')
            ->line('Payment Details:')
            ->line('• Amount: ' . $this->payment->amount . ' ' . $this->payment->currency)
            ->line('• Payment Method: ' . ($this->payment->payment_method ?? 'Unknown'))
            ->line('• User: ' . ($this->payment->user->name ?? 'Unknown'))
            ->line('• User Email: ' . ($this->payment->user->email ?? 'Unknown'))
            ->line('• Status: ' . ucfirst($this->payment->status))
            ->line('• Processing Started At: ' . $this->payment->updated_at->format('F j, Y g:i A'))
            ->action('View Payment', url('/admin/payments/' . $this->payment->id))
            ->line('Please monitor this payment for completion or failure.');
    }

    public function toDatabase(object $notifiable): array
    {
        return [
            'title' => 'Payment Processing',
            'message' => 'Payment of ' . $this->payment->amount . ' ' . $this->payment->currency . ' is being processed for ' . ($this->payment->user->name ?? 'Unknown'),
            'type' => 'payment_processing_admin',
            'data' => [
                'payment_id' => $this->payment->id,
                'amount' => $this->payment->amount,
                'currency' => $this->payment->currency,
                'payment_method' => $this->payment->payment_method,
                'user_id' => $this->payment->user->id ?? null,
                'user_name' => $this->payment->user->name ?? 'Unknown',
                'user_email' => $this->payment->user->email ?? 'Unknown',
                'status' => $this->payment->status,
                'processing_started_at' => $this->payment->updated_at->toISOString(),
            ],
            'action_url' => url('/admin/payments/' . $this->payment->id),
            'action_text' => 'View Payment',
        ];
    }
} 