<?php

namespace App\Notifications\Admin;

use App\Models\Payment;
use App\Notifications\BaseNotification;
use Illuminate\Notifications\Messages\MailMessage;

class PaymentInitiatedAdminNotification extends BaseNotification
{
    protected Payment $payment;

    public function __construct(Payment $payment)
    {
        $this->payment = $payment;
    }

    protected function buildMailMessage(object $notifiable): MailMessage
    {
        $subject = "Payment Initiated: {$this->payment->amount} {$this->payment->currency}";
        
        return (new MailMessage)
            ->subject($subject)
            ->greeting('Hello ' . ($notifiable->name ?? ''))
            ->line('A new payment has been initiated on the platform.')
            ->line('Payment Details:')
            ->line('• Amount: ' . $this->payment->amount . ' ' . $this->payment->currency)
            ->line('• Payment Method: ' . ($this->payment->payment_method ?? 'Unknown'))
            ->line('• User: ' . ($this->payment->user->name ?? 'Unknown'))
            ->line('• User Email: ' . ($this->payment->user->email ?? 'Unknown'))
            ->line('• Status: ' . ucfirst($this->payment->status))
            ->line('• Initiated At: ' . $this->payment->created_at->format('F j, Y g:i A'))
            ->action('View Payment', url('/admin/payments/' . $this->payment->id))
            ->line('Please monitor this payment for completion or failure.');
    }

    public function toDatabase(object $notifiable): array
    {
        return [
            'title' => 'Payment Initiated',
            'message' => 'Payment of ' . $this->payment->amount . ' ' . $this->payment->currency . ' initiated by ' . ($this->payment->user->name ?? 'Unknown'),
            'type' => 'payment_initiated_admin',
            'data' => [
                'payment_id' => $this->payment->id,
                'amount' => $this->payment->amount,
                'currency' => $this->payment->currency,
                'payment_method' => $this->payment->payment_method,
                'user_id' => $this->payment->user->id ?? null,
                'user_name' => $this->payment->user->name ?? 'Unknown',
                'user_email' => $this->payment->user->email ?? 'Unknown',
                'status' => $this->payment->status,
                'initiated_at' => $this->payment->created_at->toISOString(),
            ],
            'action_url' => url('/admin/payments/' . $this->payment->id),
            'action_text' => 'View Payment',
        ];
    }
}