<?php

namespace App\Notifications\Admin;

use App\Models\Payment;
use App\Notifications\BaseNotification;
use Illuminate\Notifications\Messages\MailMessage;

class PaymentDisputeAdminNotification extends BaseNotification
{
    protected Payment $payment;

    public function __construct(Payment $payment)
    {
        $this->payment = $payment;
    }

    protected function buildMailMessage(object $notifiable): MailMessage
    {
        $subject = "Payment Dispute: {$this->payment->amount} {$this->payment->currency}";
        
        return (new MailMessage)
            ->subject($subject)
            ->greeting('Hello ' . ($notifiable->name ?? ''))
            ->line('A payment dispute has been raised on the platform.')
            ->line('Payment Details:')
            ->line('• Amount: ' . $this->payment->amount . ' ' . $this->payment->currency)
            ->line('• Payment Method: ' . ($this->payment->payment_method ?? 'Unknown'))
            ->line('• User: ' . ($this->payment->user->name ?? 'Unknown'))
            ->line('• User Email: ' . ($this->payment->user->email ?? 'Unknown'))
            ->line('• Status: ' . ucfirst($this->payment->status))
            ->line('• Dispute Raised At: ' . $this->payment->updated_at->format('F j, Y g:i A'))
            ->action('View Payment', url('/admin/payments/' . $this->payment->id))
            ->line('⚠️ This dispute requires immediate attention and investigation.');
    }

    public function toDatabase(object $notifiable): array
    {
        return [
            'title' => 'Payment Dispute',
            'message' => 'Payment dispute raised for ' . $this->payment->amount . ' ' . $this->payment->currency . ' by ' . ($this->payment->user->name ?? 'Unknown'),
            'type' => 'payment_dispute_admin',
            'data' => [
                'payment_id' => $this->payment->id,
                'amount' => $this->payment->amount,
                'currency' => $this->payment->currency,
                'payment_method' => $this->payment->payment_method,
                'user_id' => $this->payment->user->id ?? null,
                'user_name' => $this->payment->user->name ?? 'Unknown',
                'user_email' => $this->payment->user->email ?? 'Unknown',
                'status' => $this->payment->status,
                'dispute_raised_at' => $this->payment->updated_at->toISOString(),
            ],
            'action_url' => url('/admin/payments/' . $this->payment->id),
            'action_text' => 'View Payment',
            'priority' => 'high',
        ];
    }
} 