<?php

namespace App\Notifications\Admin;

use App\Models\GameMatch;
use App\Notifications\BaseNotification;
use Illuminate\Notifications\Messages\MailMessage;

class MatchStartedAdminNotification extends BaseNotification
{
	protected GameMatch $match;

	public function __construct(GameMatch $match)
	{
		$this->match = $match;
	}

	protected function buildMailMessage(object $notifiable): MailMessage
	{
		$subject = "Match Started: {$this->match->team1->name} vs {$this->match->team2->name}";
		
		return (new MailMessage)
			->subject($subject)
			->greeting("Hello {$notifiable->name},")
			->line("A match has started and may require monitoring.")
			->line("Match Details:")
			->line("• Teams: {$this->match->team1->name} vs {$this->match->team2->name}")
			->line("• Tournament: {$this->match->tournament->name}")
			->line("• Start Time: {$this->match->start_time->format('F j, Y g:i A')}")
			->line("• Venue: {$this->match->venue}")
			->action('Monitor Match', route('admin.matches.show', $this->match));
	}

	public function toDatabase(object $notifiable): array
	{
		return [
			'type' => 'match_started_admin',
			'title' => "Match Started",
			'message' => "Match started: {$this->match->team1->name} vs {$this->match->team2->name}",
			'match_id' => $this->match->id,
			'team1_name' => $this->match->team1->name,
			'team2_name' => $this->match->team2->name,
			'tournament_name' => $this->match->tournament->name,
			'start_time' => $this->match->start_time->toISOString(),
			'venue' => $this->match->venue,
			'action_url' => route('admin.matches.show', $this->match),
			'created_at' => now()->toISOString(),
		];
	}
}