<?php

namespace App\Notifications\Admin;

use App\Models\GameMatch;
use App\Notifications\BaseNotification;
use Illuminate\Notifications\Messages\MailMessage;

class MatchRescheduledAdminNotification extends BaseNotification
{
	protected GameMatch $match;
	protected \DateTime $oldStartTime;
	protected ?string $reason;

	public function __construct(GameMatch $match, \DateTime $oldStartTime, ?string $reason = null)
	{
		$this->match = $match;
		$this->oldStartTime = $oldStartTime;
		$this->reason = $reason;
	}

	protected function buildMailMessage(object $notifiable): MailMessage
	{
		$subject = "Match Rescheduled: {$this->match->team1->name} vs {$this->match->team2->name}";
		
		$mailMessage = (new MailMessage)
			->subject($subject)
			->greeting("Hello {$notifiable->name},")
			->line("A match has been rescheduled.")
			->line("Match Details:")
			->line("• Teams: {$this->match->team1->name} vs {$this->match->team2->name}")
			->line("• Tournament: {$this->match->tournament->name}")
			->line("• Previous time: {$this->oldStartTime->format('F j, Y g:i A')}")
			->line("• New time: {$this->match->start_time->format('F j, Y g:i A')}")
			->line("• Venue: {$this->match->venue}");

		if ($this->reason) {
			$mailMessage->line("• Reason: {$this->reason}");
		}

		$mailMessage->action('Review Match', route('admin.matches.show', $this->match));

		return $mailMessage;
	}

	public function toDatabase(object $notifiable): array
	{
		return [
			'type' => 'match_rescheduled_admin',
			'title' => "Match Rescheduled",
			'message' => "Match rescheduled: {$this->match->team1->name} vs {$this->match->team2->name}",
			'match_id' => $this->match->id,
			'team1_name' => $this->match->team1->name,
			'team2_name' => $this->match->team2->name,
			'tournament_name' => $this->match->tournament->name,
			'old_start_time' => $this->oldStartTime->format('c'),
			'new_start_time' => $this->match->start_time->toISOString(),
			'venue' => $this->match->venue,
			'reason' => $this->reason,
			'action_url' => route('admin.matches.show', $this->match),
			'created_at' => now()->toISOString(),
		];
	}
} 