<?php

namespace App\Models;

use Illuminate\Database\Eloquent\Factories\HasFactory;
use Illuminate\Database\Eloquent\Model;
use Illuminate\Support\Facades\Cache;

class UserNotificationPreference extends Model
{
    use HasFactory;

    protected $fillable = [
        'user_id',
        'notification_type',
        'preference_type',
        'enabled',
    ];

    protected $casts = [
        'enabled' => 'boolean',
    ];

    /**
     * Get the user that owns this preference
     */
    public function user()
    {
        return $this->belongsTo(User::class);
    }

    /**
     * Check if this preference is enabled
     */
    public function isEnabled(): bool
    {
        return $this->enabled;
    }

    /**
     * Set a user's notification preference
     */
    public static function setPreference(int $userId, string $notificationType, string $preferenceType, bool $enabled): self
    {
        $preference = self::updateOrCreate(
            [
                'user_id' => $userId,
                'notification_type' => $notificationType,
                'preference_type' => $preferenceType,
            ],
            [
                'enabled' => $enabled,
            ]
        );

        // Clear cache for this user
        self::clearUserCache($userId);

        return $preference;
    }

    /**
     * Get all preferences for a user grouped by type
     */
    public static function getUserPreferences(int $userId): array
    {
        return Cache::remember("user_{$userId}_notification_preferences", 3600, function () use ($userId) {
            $preferences = self::where('user_id', $userId)->get();
            
            return [
                'categories' => $preferences->where('preference_type', 'category')->keyBy('notification_type')->toArray(),
                'individual' => $preferences->where('preference_type', 'individual')->keyBy('notification_type')->toArray(),
            ];
        });
    }

    /**
     * Check if a user has enabled a specific notification
     * Checks both category and individual preferences
     */
    public static function isEnabledForUser(int $userId, string $notificationType, string $category): bool
    {
        $cacheKey = "user_{$userId}_notification_{$notificationType}_enabled";
        
        return Cache::remember($cacheKey, 3600, function () use ($userId, $notificationType, $category) {
            // First check if individual notification is explicitly set
            $individualPref = self::where('user_id', $userId)
                ->where('notification_type', $notificationType)
                ->where('preference_type', 'individual')
                ->first();

            if ($individualPref) {
                return $individualPref->enabled;
            }

            // Then check category preference
            $categoryPref = self::where('user_id', $userId)
                ->where('notification_type', $category)
                ->where('preference_type', 'category')
                ->first();

            if ($categoryPref) {
                return $categoryPref->enabled;
            }

            // Default to enabled if no preference is set
            return true;
        });
    }

    /**
     * Bulk update preferences for a user
     */
    public static function bulkUpdate(int $userId, array $preferences): void
    {
        foreach ($preferences as $pref) {
            self::setPreference(
                $userId,
                $pref['notification_type'],
                $pref['preference_type'],
                $pref['enabled']
            );
        }
    }

    /**
     * Clear cache for a specific user
     */
    public static function clearUserCache(int $userId): void
    {
        Cache::forget("user_{$userId}_notification_preferences");
        
        // Clear individual notification caches for this user
        // This could be optimized by storing a list of keys, but for now we'll rely on TTL
    }

    /**
     * Boot method to clear cache on changes
     */
    protected static function boot()
    {
        parent::boot();

        static::saved(function ($preference) {
            self::clearUserCache($preference->user_id);
        });

        static::deleted(function ($preference) {
            self::clearUserCache($preference->user_id);
        });
    }
}

