<?php

namespace App\Models;

use Illuminate\Contracts\Auth\MustVerifyEmail;
use Illuminate\Database\Eloquent\Factories\HasFactory;
use Illuminate\Database\Eloquent\Relations\HasMany;
use Illuminate\Database\Eloquent\Relations\BelongsToMany;
use Illuminate\Foundation\Auth\User as Authenticatable;
use Illuminate\Notifications\Notifiable;

class User extends Authenticatable implements MustVerifyEmail
{
    /** @use HasFactory<\Database\Factories\UserFactory> */
    use HasFactory, Notifiable;

    /**
     * The attributes that are mass assignable.
     *
     * @var list<string>
     */
    protected $fillable = [
        'name',
        'email',
        'password',
        'phone',
        'profile_complete',
        'payment_status',
        'user_type', // team, player, official
        'otp_code',
        'otp_expires_at',
        'must_change_password',
        'terms_accepted_at',
        // KYC fields
        'id_number',
        'passport_number',
        'date_of_birth',
        'gender',
        'nationality',
        'county',
        'sub_county',
        'ward',
        'address',
        'postal_code',
        'occupation',
        'employer',
        'emergency_contact_name',
        'emergency_contact_phone',
        'emergency_contact_relationship',
        'profile_image',
        'bio',
        'kyc_documents',
        'kyc_status',
        'kyc_verified_at',
        'kyc_notes',
    ];

    /**
     * The attributes that should be hidden for serialization.
     *
     * @var list<string>
     */
    protected $hidden = [
        'password',
        'remember_token',
    ];

    /**
     * Get the attributes that should be cast.
     *
     * @return array<string, string>
     */
    protected function casts(): array
    {
        return [
            'email_verified_at' => 'datetime',
            'password' => 'hashed',
            'profile_complete' => 'boolean',
            'otp_expires_at' => 'datetime',
            'must_change_password' => 'boolean',
            'terms_accepted_at' => 'datetime',
            'date_of_birth' => 'date',
            'kyc_documents' => 'array',
            'kyc_verified_at' => 'datetime',
        ];
    }

    public function players(): HasMany
    {
        return $this->hasMany(Player::class);
    }

    public function teams(): HasMany
    {
        return $this->hasMany(Team::class, 'captain_id', 'id');
    }

    public function managedTeams(): HasMany
    {
        return $this->hasMany(Team::class, 'captain_id', 'id');
    }

    public function teamMemberships(): BelongsToMany
    {
        return $this->belongsToMany(Team::class, 'team_members')
            ->withPivot('role')
            ->withTimestamps();
    }

    public function roles(): BelongsToMany
    {
        return $this->belongsToMany(Role::class)
                    ->withPivot(['is_primary', 'assigned_at', 'expires_at', 'metadata'])
                    ->withTimestamps();
    }

    public function emailCredentials(): BelongsToMany
    {
        return $this->belongsToMany(EmailCredential::class, 'email_user')
                    ->withPivot(['can_send', 'can_receive', 'permissions', 'granted_at', 'expires_at'])
                    ->withTimestamps();
    }

    // Role-related methods
    public function hasRole(string $roleName): bool
    {
        return $this->roles()->where('name', $roleName)->exists();
    }

    public function hasAnyRole(array $roleNames): bool
    {
        return $this->roles()->whereIn('name', $roleNames)->exists();
    }

    public function hasAllRoles(array $roleNames): bool
    {
        $userRoleNames = $this->roles()->pluck('name')->toArray();
        return count(array_intersect($roleNames, $userRoleNames)) === count($roleNames);
    }

    public function hasPermission(string $permission): bool
    {
        return $this->roles()
                    ->where('is_active', true)
                    ->get()
                    ->some(fn($role) => $role->hasPermission($permission));
    }

    public function hasAnyPermission(array $permissions): bool
    {
        return $this->roles()
                    ->where('is_active', true)
                    ->get()
                    ->some(fn($role) => $role->hasAnyPermission($permissions));
    }

    public function hasAllPermissions(array $permissions): bool
    {
        $activeRoles = $this->roles()->where('is_active', true)->get();
        
        foreach ($permissions as $permission) {
            $hasPermission = $activeRoles->some(fn($role) => $role->hasPermission($permission));
            if (!$hasPermission) {
                return false;
            }
        }
        return true;
    }

    public function getAllPermissions()
    {
        return $this->roles()
                    ->where('is_active', true)
                    ->get()
                    ->flatMap(fn($role) => $role->permissions()->pluck('name'))
                    ->unique()
                    ->values();
    }   

    public function assignRole(Role $role, bool $isPrimary = false): void
    {
        $this->roles()->attach($role->id, [
            'is_primary' => $isPrimary,
            'assigned_at' => now(),
        ]);
    }

    public function removeRole(Role $role): void
    {
        $this->roles()->detach($role->id);
    }

    public function syncRoles(array $roleIds): void
    {
        $this->roles()->sync($roleIds);
    }

    public function getPrimaryRole(): ?Role
    {
        return $this->roles()
                    ->wherePivot('is_primary', true)
                    ->first();
    }

    public function getActiveRoles(): \Illuminate\Database\Eloquent\Collection
    {
        return $this->roles()
                    ->where('is_active', true)
                    ->get();
    }

    public function isAdmin(): bool
    {
        return $this->hasRole('admin');
    }

    public function isReferee(): bool
    {
        return $this->hasRole('referee');
    }

    public function isPlayer(): bool
    {
        return $this->hasRole('player');
    }

    public function isOrganizer(): bool
    {
        return $this->hasRole('organizer');
    }

    // Payment relationship
    public function payments(): HasMany
    {
        return $this->hasMany(Payment::class);
    }

    /**
     * Get the user's notification preferences
     */
    public function notificationPreferences(): HasMany
    {
        return $this->hasMany(UserNotificationPreference::class);
    }

    /**
     * Send the password reset notification.
     * 
     * Sends synchronously for immediate delivery.
     * Use fast SMTP service (Mailgun/SendGrid) for best speed.
     *
     * @param  string  $token
     * @return void
     */
    public function sendPasswordResetNotification($token)
    {
        $this->notify(new \App\Notifications\Auth\ResetPassword($token));
    }

    /**
     * Send the email verification notification.
     * 
     * Sends synchronously for immediate delivery.
     * Use fast SMTP service (Mailgun/SendGrid) for best speed.
     *
     * @return void
     */
    public function sendEmailVerificationNotification()
    {
        $this->notify(new \App\Notifications\Auth\VerifyEmail());
    }
}
