<?php

namespace App\Models;

use Illuminate\Database\Eloquent\Factories\HasFactory;
use Illuminate\Database\Eloquent\Model;
use Illuminate\Database\Eloquent\Relations\HasMany;
use Illuminate\Database\Eloquent\Relations\BelongsToMany;
use Illuminate\Database\Eloquent\Relations\BelongsTo;
use Illuminate\Database\Eloquent\Builder;
use Illuminate\Support\Str;
use Carbon\Carbon;

class Tournament extends Model
{
    use HasFactory;

    protected $table = 'tournaments';   
    protected $fillable = [
        'name',
        'slug',
        'description',
        'location',
        'venue',
        'organizer',
        'contact_email',
        'contact_phone',
        'website',
        'type',
        'status',
        'start_date',
        'end_date',
        'registration_start',
        'registration_end',
        'max_participants',
        'min_participants',
        'min_players_per_team',
        'max_players_per_team',
        'auto_accept_registrations',
        'registration_status',
        'game_title',
        'game_version',
        'format',
        'team_size',
        'rounds',
        'matches_per_round',
        'bracket_settings',
        'stream_url',
        'youtube_url',
        'twitch_url',
        'is_streamed',
        'is_featured',
        'entry_fee',
        'prize_pool',
        'prize_distribution',
        'currency',
        'sponsor_info',
        'rewards',
        'total_participants',
        'total_matches',
        'completed_matches',
        'average_match_duration',
        'tournament_stats',
        'banner_image',
        'logo_image',
        'social_links',
        'announcement',
        'visibility',
        'allow_spectators',
        'require_approval',
        'enable_two_tier_approval',
        'first_approver_id',
        'second_approver_id',
        'approval_settings',
        'enable_categories',
        'category_fees',
        'available_categories',
        'admin_notes',
        'metadata',
        'rules',
        'settings',
        'published_at',
        'completed_at',
        'cancelled_at',
    ];

    protected $casts = [
        'start_date' => 'datetime',
        'end_date' => 'datetime',
        'registration_start' => 'datetime',
        'registration_end' => 'datetime',
        'published_at' => 'datetime',
        'completed_at' => 'datetime',
        'cancelled_at' => 'datetime',
        'auto_accept_registrations' => 'boolean',
        'is_streamed' => 'boolean',
        'is_featured' => 'boolean',
        'allow_spectators' => 'boolean',
        'require_approval' => 'boolean',
        'enable_two_tier_approval' => 'boolean',
        'approval_settings' => 'array',
        'enable_categories' => 'boolean',
        'category_fees' => 'array',
        'available_categories' => 'array',
        'entry_fee' => 'decimal:2',
        'prize_pool' => 'decimal:2',
        'average_match_duration' => 'decimal:2',
        'rules' => 'array',
        'settings' => 'array',
        'bracket_settings' => 'array',
        'prize_distribution' => 'array',
        'sponsor_info' => 'array',
        'rewards' => 'array',
        'tournament_stats' => 'array',
        'social_links' => 'array',
        'admin_notes' => 'array',
        'metadata' => 'array',
    ];

    protected $dates = [
        'start_date',
        'end_date',
        'registration_start',
        'registration_end',
        'published_at',
        'completed_at',
        'cancelled_at',
    ];

    // Relationships
    public function players(): HasMany
    {
        return $this->hasMany(Player::class);
    }

    public function teams(): BelongsToMany
    {
        return $this->belongsToMany(Team::class, 'team_tournament')
            ->withPivot('status', 'registered_at', 'approved_at', 'registration_notes', 'tournament_specific_data', 
                       'approval_status', 'first_approved_by', 'first_approved_at', 'second_approved_by', 
                       'second_approved_at', 'first_approval_notes', 'second_approval_notes', 
                       'rejection_reason', 'rejected_by', 'rejected_at')
            ->withTimestamps();
    }

    public function matches(): HasMany
    {
        return $this->hasMany(GameMatch::class);
    }

    public function brackets(): HasMany
    {
        return $this->hasMany(Bracket::class);
    }

    public function firstApprover(): BelongsTo
    {
        return $this->belongsTo(User::class, 'first_approver_id');
    }

    public function secondApprover(): BelongsTo
    {
        return $this->belongsTo(User::class, 'second_approver_id');
    }

    // New methods for multi-tournament team support
    public function getRegisteredTeams()
    {
        return $this->teams()->wherePivot('status', 'registered')->get();
    }

    public function getActiveTeams()
    {
        return $this->teams()->wherePivot('status', 'active')->get();
    }

    public function getEliminatedTeams()
    {
        return $this->teams()->wherePivot('status', 'eliminated')->get();
    }

    public function getWithdrawnTeams()
    {
        return $this->teams()->wherePivot('status', 'withdrawn')->get();
    }

    public function isTeamRegistered(Team $team): bool
    {
        return $this->teams()->where('team_id', $team->id)->exists();
    }

    public function getTeamStatus(Team $team): ?string
    {
        $pivot = $this->teams()->where('team_id', $team->id)->first();
        return $pivot ? $pivot->pivot->status : null;
    }

    public function approveTeam(Team $team): void
    {
        $this->teams()->updateExistingPivot($team->id, [
            'status' => 'active',
            'approved_at' => now(),
        ]);
    }

    public function rejectTeam(Team $team): void
    {
        $this->teams()->detach($team->id);
    }

    // Two-tier approval methods
    public function firstApproveTeam(Team $team, User $approver, string $notes = null): void
    {
        $this->teams()->updateExistingPivot($team->id, [
            'approval_status' => 'first_approved',
            'first_approved_by' => $approver->id,
            'first_approved_at' => now(),
            'first_approval_notes' => $notes,
        ]);
    }

    public function secondApproveTeam(Team $team, User $approver, string $notes = null): void
    {
        $this->teams()->updateExistingPivot($team->id, [
            'approval_status' => 'second_approved',
            'status' => 'active',
            'second_approved_by' => $approver->id,
            'second_approved_at' => now(),
            'second_approval_notes' => $notes,
            'approved_at' => now(),
        ]);
    }

    public function rejectTeamApproval(Team $team, User $rejector, string $reason): void
    {
        $this->teams()->updateExistingPivot($team->id, [
            'approval_status' => 'rejected',
            'rejected_by' => $rejector->id,
            'rejected_at' => now(),
            'rejection_reason' => $reason,
        ]);
    }

    public function getPendingFirstApprovalTeams()
    {
        return $this->teams()->wherePivot('approval_status', 'pending')->get();
    }

    public function getFirstApprovedTeams()
    {
        return $this->teams()->wherePivot('approval_status', 'first_approved')->get();
    }

    public function getSecondApprovedTeams()
    {
        return $this->teams()->wherePivot('approval_status', 'second_approved')->get();
    }

    public function getRejectedTeams()
    {
        return $this->teams()->wherePivot('approval_status', 'rejected')->get();
    }

    public function canUserApprove(User $user): bool
    {
        // Super admin can always approve
        if ($user->hasRole('super_admin')) {
            return true;
        }

        // Regular admin can approve if no two-tier system or if they are assigned approvers
        if ($user->hasRole('admin')) {
            return true;
        }

        if ($this->enable_two_tier_approval) {
            return $user->id === $this->first_approver_id || $user->id === $this->second_approver_id;
        }

        return false;
    }

    public function canUserFirstApprove(User $user): bool
    {
        // Super admin can always approve
        if ($user->hasRole('super_admin')) {
            return true;
        }

        // Regular admin can approve if no two-tier system
        if ($user->hasRole('admin')) {
            return true;
        }

        return $this->enable_two_tier_approval && $user->id === $this->first_approver_id;
    }

    public function canUserSecondApprove(User $user): bool
    {
        // Super admin can always approve
        if ($user->hasRole('super_admin')) {
            return true;
        }

        // Regular admin can approve if no two-tier system
        if ($user->hasRole('admin')) {
            return true;
        }

        return $this->enable_two_tier_approval && $user->id === $this->second_approver_id;
    }

    /**
     * Get the default first approver from environment or tournament settings
     */
    public function getDefaultFirstApprover(): ?User
    {
        // First check tournament-specific setting
        if ($this->first_approver_id) {
            return User::find($this->first_approver_id);
        }

        // Fall back to environment variable
        $envApproverId = env('TOURNAMENT_DEFAULT_FIRST_APPROVER_ID');
        if ($envApproverId) {
            return User::find($envApproverId);
        }

        return null;
    }

    /**
     * Get the default second approver from environment or tournament settings
     */
    public function getDefaultSecondApprover(): ?User
    {
        // First check tournament-specific setting
        if ($this->second_approver_id) {
            return User::find($this->second_approver_id);
        }

        // Fall back to environment variable
        $envApproverId = env('TOURNAMENT_DEFAULT_SECOND_APPROVER_ID');
        if ($envApproverId) {
            return User::find($envApproverId);
        }

        return null;
    }

    /**
     * Check if user can approve a specific team based on approval status
     */
    public function canUserApproveTeam(User $user, Team $team): array
    {
        $teamApprovalStatus = $team->getApprovalStatus($this);
        
        $canApprove = false;
        $canFirstApprove = false;
        $canSecondApprove = false;
        $canReject = false;
        $reason = '';

        // Super admin can do everything
        if ($user->hasRole('super_admin')) {
            return [
                'can_approve' => true,
                'can_first_approve' => true,
                'can_second_approve' => true,
                'can_reject' => true,
                'is_super_admin' => true,
                'reason' => 'Super admin override'
            ];
        }

        // Regular admin can do everything if no two-tier system
        if ($user->hasRole('admin') && !$this->enable_two_tier_approval) {
            return [
                'can_approve' => true,
                'can_first_approve' => true,
                'can_second_approve' => true,
                'can_reject' => true,
                'is_admin' => true,
                'reason' => 'Admin override'
            ];
        }

        // Two-tier approval logic
        if ($this->enable_two_tier_approval) {
            switch ($teamApprovalStatus) {
                case 'pending':
                    $canFirstApprove = $this->canUserFirstApprove($user);
                    $canReject = $this->canUserApprove($user);
                    $reason = $canFirstApprove ? 'Can perform first approval' : 'Not authorized for first approval';
                    break;
                    
                case 'first_approved':
                    $canSecondApprove = $this->canUserSecondApprove($user);
                    $canReject = $this->canUserApprove($user);
                    $reason = $canSecondApprove ? 'Can perform second approval' : 'Not authorized for second approval';
                    break;
                    
                case 'second_approved':
                case 'active':
                    $reason = 'Team already fully approved';
                    break;
                    
                case 'rejected':
                    $reason = 'Team has been rejected';
                    break;
            }
        } else {
            // Single approval system
            $canApprove = $this->canUserApprove($user);
            $canReject = $this->canUserApprove($user);
            $reason = $canApprove ? 'Can approve team' : 'Not authorized to approve';
        }

        return [
            'can_approve' => $canApprove,
            'can_first_approve' => $canFirstApprove,
            'can_second_approve' => $canSecondApprove,
            'can_reject' => $canReject,
            'reason' => $reason,
            'team_status' => $teamApprovalStatus,
            'is_two_tier' => $this->enable_two_tier_approval,
        ];
    }

    // Scopes
    public function scopeActive(Builder $query): void
    {
        $query->where('status', 'active');
    }

    public function scopePublished(Builder $query): void
    {
        $query->where('visibility', 'public')
              ->whereNotNull('published_at')
              ->where('published_at', '<=', now());
    }

    public function scopeFeatured(Builder $query): void
    {
        $query->where('is_featured', true);
    }

    public function scopeUpcoming(Builder $query): void
    {
        $query->where('start_date', '>', now());
    }

    public function scopeOngoing(Builder $query): void
    {
        $query->where('start_date', '<=', now())
              ->where('end_date', '>=', now())
              ->where('status', 'active');
    }

    public function scopeCompleted(Builder $query): void
    {
        $query->where('status', 'completed');
    }

    public function scopeRegistrationOpen(Builder $query): void
    {
        $query->where('registration_status', 'open')
              ->where(function($q) {
                  $q->whereNull('registration_start')
                    ->orWhere('registration_start', '<=', now());
              })
              ->where(function($q) {
                  $q->whereNull('registration_end')
                    ->orWhere('registration_end', '>=', now());
              });
    }

    // Mutators
    public function setNameAttribute($value)
    {
        $this->attributes['name'] = $value;
    }

    // Accessors
    public function getIsRegistrationOpenAttribute(): bool
    {
        if ($this->registration_status !== 'open') {
            return false;
        }

        $now = now();
        
        if ($this->registration_start && $this->registration_start > $now) {
            return false;
        }
        
        if ($this->registration_end && $this->registration_end < $now) {
            return false;
        }

        return true;
    }

    public function getIsFullAttribute(): bool
    {
        return $this->max_participants && $this->total_participants >= $this->max_participants;
    }

    public function getProgressPercentageAttribute(): float
    {
        if ($this->total_matches === 0) {
            return 0;
        }

        return round(($this->completed_matches / $this->total_matches) * 100, 2);
    }

    public function getDurationAttribute(): ?int
    {
        if (!$this->start_date || !$this->end_date) {
            return null;
        }

        return $this->start_date->diffInDays($this->end_date);
    }

    public function getStatusColorAttribute(): string
    {
        return match($this->status) {
            'draft' => 'bg-gray-100 text-gray-800',
            'active' => 'bg-green-100 text-green-800',
            'completed' => 'bg-blue-100 text-blue-800',
            'cancelled' => 'bg-red-100 text-red-800',
            default => 'bg-gray-100 text-gray-800',
        };
    }

    public function getFormatDisplayAttribute(): string
    {
        return match($this->format) {
            'individual' => 'Individual',
            'team' => 'Team',
            'mixed' => 'Mixed',
            default => 'Individual',
        };
    }

    // Advanced Methods
    public function canRegister(): bool
    {
        return $this->is_registration_open && !$this->is_full;
    }

    public function updateParticipantCount(): void
    {
        $count = $this->format === 'team' 
            ? $this->teams()->count() 
            : $this->players()->count();
            
        $this->update(['total_participants' => $count]);
    }

    public function updateMatchStats(): void
    {
        $totalMatches = $this->matches()->count();
        $completedMatches = $this->matches()->where('status', 'completed')->count();
        $avgDuration = $this->matches()
            ->where('status', 'completed')
            ->whereNotNull('started_at')
            ->whereNotNull('completed_at')
            ->get()
            ->avg(function ($match) {
                return $match->started_at->diffInMinutes($match->completed_at);
            });

        $this->update([
            'total_matches' => $totalMatches,
            'completed_matches' => $completedMatches,
            'average_match_duration' => $avgDuration ? round($avgDuration, 2) : null,
        ]);
    }

    public function generateBracket(): void
    {
        // Implementation for bracket generation based on tournament type
        // This would be expanded based on specific tournament logic
    }

    public function getDetailedStats(): array
    {
        return [
            'participants' => [
                'total' => $this->total_participants,
                'registered' => $this->players()->where('status', 'registered')->count(),
                'active' => $this->players()->where('status', 'active')->count(),
                'eliminated' => $this->players()->where('status', 'eliminated')->count(),
            ],
            'matches' => [
                'total' => $this->total_matches,
                'completed' => $this->completed_matches,
                'pending' => $this->matches()->where('status', 'pending')->count(),
                'in_progress' => $this->matches()->where('status', 'in_progress')->count(),
            ],
            'progress' => [
                'percentage' => $this->progress_percentage,
                'duration_days' => $this->duration,
                'days_remaining' => $this->end_date ? max(0, now()->diffInDays($this->end_date, false)) : null,
            ],
            'financial' => [
                'entry_fee' => $this->entry_fee,
                'prize_pool' => $this->prize_pool,
                'total_collected' => $this->entry_fee * $this->total_participants,
            ],
        ];
    }

    public function getSocialLinks(): array
    {
        $links = $this->social_links ?? [];
        
        // Add streaming links if available
        if ($this->stream_url) {
            $links['stream'] = $this->stream_url;
        }
        if ($this->youtube_url) {
            $links['youtube'] = $this->youtube_url;
        }
        if ($this->twitch_url) {
            $links['twitch'] = $this->twitch_url;
        }
        if ($this->website) {
            $links['website'] = $this->website;
        }

        return $links;
    }

    public function publish(): void
    {
        $this->update([
            'visibility' => 'public',
            'published_at' => now(),
        ]);
    }

    public function complete(): void
    {
        $this->update([
            'status' => 'completed',
            'completed_at' => now(),
        ]);
    }

    public function cancel(): void
    {
        $this->update([
            'status' => 'cancelled',
            'cancelled_at' => now(),
        ]);
    }

    /**
     * Get entry fee for a specific category
     */
    public function getFeeForCategory(string $category): float
    {
        // If categories not enabled, use default entry_fee
        if (!$this->enable_categories) {
            \Log::info('Categories not enabled, using tournament entry fee', [
                'tournament_id' => $this->id,
                'entry_fee' => $this->entry_fee
            ]);
            return (float) ($this->entry_fee ?? 0);
        }

        // Check category_fees JSON for specific category
        $categoryFees = $this->category_fees ?? [];
        
        if (isset($categoryFees[$category]) && $categoryFees[$category] > 0) {
            \Log::info('Using category-specific fee', [
                'tournament_id' => $this->id,
                'category' => $category,
                'fee' => $categoryFees[$category]
            ]);
            return (float) $categoryFees[$category];
        }

        // Fall back to default from config
        $categories = config('tournament.categories', []);
        if (isset($categories[$category]['default_fee'])) {
            \Log::info('Using config default fee for category', [
                'tournament_id' => $this->id,
                'category' => $category,
                'fee' => $categories[$category]['default_fee']
            ]);
            return (float) $categories[$category]['default_fee'];
        }

        // Ultimate fallback to tournament entry fee
        \Log::info('Using tournament entry fee as fallback', [
            'tournament_id' => $this->id,
            'category' => $category,
            'entry_fee' => $this->entry_fee
        ]);
        return (float) ($this->entry_fee ?? 0);
    }

    /**
     * Get available categories for this tournament
     */
    public function getAvailableCategories(): array
    {
        if (!$this->enable_categories) {
            return ['men']; // Default to men only if categories disabled
        }

        $available = $this->available_categories ?? [];
        
        // If no categories specified, all are available
        if (empty($available)) {
            return array_keys(config('tournament.categories', []));
        }

        return $available;
    }

    /**
     * Check if a category is enabled for this tournament
     */
    public function isCategoryEnabled(string $category): bool
    {
        if (!$this->enable_categories) {
            return $category === 'men'; // Only men category if categories disabled
        }

        $availableCategories = $this->getAvailableCategories();
        return in_array($category, $availableCategories);
    }

    /**
     * Get category fees for all available categories
     */
    public function getAllCategoryFees(): array
    {
        $availableCategories = $this->getAvailableCategories();
        $fees = [];

        foreach ($availableCategories as $category) {
            $fees[$category] = $this->getFeeForCategory($category);
        }

        return $fees;
    }

    /**
     * Get team count per category
     */
    public function getCategoryStatistics(): array
    {
        if (!$this->enable_categories) {
            return [];
        }

        return $this->teams()
            ->select('category', \DB::raw('count(*) as count'))
            ->groupBy('category')
            ->pluck('count', 'category')
            ->toArray();
    }
}

