<?php

namespace App\Models;

use Illuminate\Database\Eloquent\Factories\HasFactory;
use Illuminate\Database\Eloquent\Model;
use Illuminate\Database\Eloquent\Relations\BelongsTo;
use Illuminate\Database\Eloquent\Relations\BelongsToMany;
use Illuminate\Database\Eloquent\Relations\HasMany;

class Team extends Model
{
    use HasFactory;

    protected $fillable = [
        'name',
        'description',
        'logo',
        'captain_id',
        'country',
        'county',
        'location',
        'stats',
        'status',
        'category', // Team category: men, women, youth_male, youth_female, children
        // Payment fields
        'payment_status',
        'transaction_id',
        'payment_amount',
        'payment_method',
        'payment_completed_at',
        // Additional settings
        'max_players',
        'notes',
        'notify_match_updates',
        'notify_tournament_updates',
    ];

    protected $casts = [
        'stats' => 'array',
        'payment_completed_at' => 'datetime',
        'payment_amount' => 'decimal:2',
        'notify_match_updates' => 'boolean',
        'notify_tournament_updates' => 'boolean',
    ];

    // Many-to-many relationship with tournaments
    public function tournaments(): BelongsToMany
    {
        return $this->belongsToMany(Tournament::class, 'team_tournament')
            ->withPivot('status', 'registered_at', 'approved_at', 'registration_notes', 'tournament_specific_data',
                       'approval_status', 'first_approved_by', 'first_approved_at', 'second_approved_by', 
                       'second_approved_at', 'first_approval_notes', 'second_approval_notes', 
                       'rejection_reason', 'rejected_by', 'rejected_at')
            ->withTimestamps();
    }

    // Get the current active tournament (for backward compatibility)
    public function tournament(): BelongsTo
    {
        return $this->belongsTo(Tournament::class);
    }

    public function payments(): HasMany
    {
        return $this->hasMany(Payment::class, 'team_id');
    }

    public function captain(): BelongsTo
    {
        return $this->belongsTo(User::class, 'captain_id');
    }

    public function manager(): BelongsTo
    {
        return $this->belongsTo(User::class, 'captain_id');
    }

    public function members(): BelongsToMany
    {
        return $this->belongsToMany(User::class, 'team_members')
            ->withPivot('role')
            ->withTimestamps();
    }

    public function matchesAsTeam1(): HasMany
    {
        return $this->hasMany(GameMatch::class, 'team1_id');
    }

    public function matchesAsTeam2(): HasMany
    {
        return $this->hasMany(GameMatch::class, 'team2_id');
    }

    public function matchesAsWinningTeam(): HasMany
    {
        return $this->hasMany(GameMatch::class, 'winning_team_id');
    }

    // New relationships for team registration
    public function teamPlayers(): HasMany
    {
        return $this->hasMany(Player::class);
    }

    public function players()
    {
        return $this->hasMany(Player::class);
    }

    public function teamOfficials(): HasMany
    {
        return $this->hasMany(TeamOfficial::class);
    }

    public function officials()
    {
        return $this->hasMany(TeamOfficial::class);
    }

    public function formations(): HasMany
    {
        return $this->hasMany(TeamFormation::class);
    }

    public function defaultFormation(): HasMany
    {
        return $this->hasMany(TeamFormation::class)->where('is_default', true);
    }

    // New methods for multi-tournament support
    public function isRegisteredForTournament(Tournament $tournament): bool
    {
        return $this->tournaments()->where('tournament_id', $tournament->id)->exists();
    }

    public function getTournamentStatus(Tournament $tournament): ?string
    {
        $pivot = $this->tournaments()->where('tournament_id', $tournament->id)->first();
        return $pivot ? $pivot->pivot->status : null;
    }

    public function registerForTournament(Tournament $tournament, array $data = []): void
    {
        $this->tournaments()->attach($tournament->id, [
            'status' => 'registered',
            'registered_at' => now(),
            'registration_notes' => $data['notes'] ?? null,
            'tournament_specific_data' => isset($data['tournament_data'])
                ? (is_array($data['tournament_data']) ? json_encode($data['tournament_data']) : $data['tournament_data'])
                : null,
        ]);
    }

    public function updateTournamentStatus(Tournament $tournament, string $status): void
    {
        $this->tournaments()->updateExistingPivot($tournament->id, [
            'status' => $status,
            'approved_at' => $status === 'active' ? now() : null,
        ]);
    }

    public function withdrawFromTournament(Tournament $tournament): void
    {
        $this->tournaments()->updateExistingPivot($tournament->id, [
            'status' => 'withdrawn',
        ]);
    }

    public function getActiveTournaments()
    {
        return $this->tournaments()->wherePivot('status', 'active')->get();
    }

    // Approval status methods
    public function getApprovalStatus(Tournament $tournament): ?string
    {
        $pivot = $this->tournaments()->where('tournament_id', $tournament->id)->first();
        return $pivot ? $pivot->pivot->approval_status : null;
    }

    public function isPendingApproval(Tournament $tournament): bool
    {
        return $this->getApprovalStatus($tournament) === 'pending';
    }

    public function isFirstApproved(Tournament $tournament): bool
    {
        return $this->getApprovalStatus($tournament) === 'first_approved';
    }

    public function isSecondApproved(Tournament $tournament): bool
    {
        return $this->getApprovalStatus($tournament) === 'second_approved';
    }

    public function isRejected(Tournament $tournament): bool
    {
        return $this->getApprovalStatus($tournament) === 'rejected';
    }

    public function getFirstApprover(Tournament $tournament): ?User
    {
        $pivot = $this->tournaments()->where('tournament_id', $tournament->id)->first();
        if ($pivot && $pivot->pivot->first_approved_by) {
            return User::find($pivot->pivot->first_approved_by);
        }
        return null;
    }

    public function getSecondApprover(Tournament $tournament): ?User
    {
        $pivot = $this->tournaments()->where('tournament_id', $tournament->id)->first();
        if ($pivot && $pivot->pivot->second_approved_by) {
            return User::find($pivot->pivot->second_approved_by);
        }
        return null;
    }

    public function getRegisteredTournaments()
    {
        return $this->tournaments()->wherePivot('status', 'registered')->get();
    }

    // Scope to get teams for a specific tournament
    public function scopeForTournament($query, ?Tournament $tournament)
    {
        if (!$tournament) {
            return $query;
        }
        
        return $query->whereHas('tournaments', function ($q) use ($tournament) {
            $q->where('tournament_id', $tournament->id);
        });
    }

    // Scope to get teams with specific status in a tournament
    public function scopeForTournamentWithStatus($query, ?Tournament $tournament, string $status)
    {
        if (!$tournament) {
            return $query;
        }
        
        return $query->whereHas('tournaments', function ($q) use ($tournament, $status) {
            $q->where('tournament_id', $tournament->id)
              ->where('status', $status);
        });
    }

    // Player completeness and approval methods
    public function getPlayerCompletenessStatus(?Tournament $tournament = null): array
    {
        // Get tournament-specific player limits
        $minPlayers = $tournament ? ($tournament->min_players_per_team ?? config('tournament.min_players_per_team', 11)) : 11;
        $maxPlayers = $tournament ? ($tournament->max_players_per_team ?? config('tournament.max_players_per_team', 23)) : 23;
        
        $players = $this->players;
        $totalPlayers = $players->count();
        $completePlayers = 0;
        $incompletePlayers = [];
        $missingRequirements = [];

        foreach ($players as $player) {
            $completeness = $this->checkPlayerCompleteness($player);
            if ($completeness['is_complete']) {
                $completePlayers++;
            } else {
                $incompletePlayers[] = [
                    'player' => $player,
                    'missing_fields' => $completeness['missing_fields'],
                    'completeness_percentage' => $completeness['completeness_percentage']
                ];
            }
        }

        return [
            'total_players' => $totalPlayers,
            'complete_players' => $completePlayers,
            'incomplete_players' => $incompletePlayers,
            'team_completeness_percentage' => $totalPlayers > 0 ? round(($completePlayers / $totalPlayers) * 100, 2) : 0,
            'is_team_complete' => $completePlayers === $totalPlayers && $totalPlayers >= $minPlayers,
            'minimum_players_met' => $totalPlayers >= $minPlayers,
            'recommended_players_met' => $totalPlayers >= $maxPlayers,
        ];
    }

    public function checkPlayerCompleteness(Player $player): array
    {
        // Required fields for player profile approval
        // Note: display_name is set by users during registration and not checked here
        $requiredFields = [
            'display_name' => 'Display Name',
            'phone' => 'Phone Number',
            'date_of_birth' => 'Date of Birth',
            'gender' => 'Gender',
            'football_position' => 'Playing Position',
            'national_id_front' => 'National ID (Front)',
            'national_id_back' => 'National ID (Back)',
            'id_number' => 'ID Number',
        ];

        $optionalFields = [
            'avatar' => 'Profile Picture',
            'bio' => 'Biography',
            'country' => 'Country',
            'city' => 'City/Location',
            'jersey_number' => 'Jersey Number',
        ];

        $missingRequired = [];
        $missingOptional = [];

        // Check required fields
        foreach ($requiredFields as $field => $label) {
            if (empty($player->$field)) {
                $missingRequired[] = $label;
            }
        }

        // Check optional fields
        foreach ($optionalFields as $field => $label) {
            if (empty($player->$field)) {
                $missingOptional[] = $label;
            }
        }

        $totalFields = count($requiredFields) + count($optionalFields);
        $completedFields = $totalFields - count($missingRequired) - count($missingOptional);
        $completenessPercentage = round(($completedFields / $totalFields) * 100, 2);

        return [
            'is_complete' => empty($missingRequired),
            'completeness_percentage' => $completenessPercentage,
            'missing_required' => $missingRequired,
            'missing_optional' => $missingOptional,
            'missing_fields' => array_merge($missingRequired, $missingOptional),
            'verification_status' => $player->verification_status,
            'approval_status' => $player->approval_status ?? 'pending',
        ];
    }

    public function canBeApprovedForTournament(Tournament $tournament): array
    {
        $completeness = $this->getPlayerCompletenessStatus($tournament);
        $teamStatus = $this->getApprovalStatus($tournament);
        
        // Get tournament-specific minimum players
        $minPlayers = $tournament->min_players_per_team ?? config('tournament.min_players_per_team', 11);
        
        $canApprove = true;
        $reasons = [];

        // Check team registration status
        if (!$this->isRegisteredForTournament($tournament)) {
            $canApprove = false;
            $reasons[] = 'Team is not registered for this tournament';
        }

        // Check minimum players requirement
        if (!$completeness['minimum_players_met']) {
            $canApprove = false;
            $reasons[] = 'Team must have at least ' . $minPlayers . ' players (currently has ' . $completeness['total_players'] . ')';
        }

        // Check if all players are complete
        if (!$completeness['is_team_complete']) {
            $canApprove = false;
            $reasons[] = 'All players must have complete profiles';
        }

        // Check if team is already approved
        if ($teamStatus === 'second_approved' || $teamStatus === 'active') {
            $canApprove = false;
            $reasons[] = 'Team is already approved for this tournament';
        }

        // Check if team is rejected
        if ($teamStatus === 'rejected') {
            $canApprove = false;
            $reasons[] = 'Team has been rejected and cannot be approved';
        }

        return [
            'can_approve' => $canApprove,
            'reasons' => $reasons,
            'completeness' => $completeness,
            'team_status' => $teamStatus,
        ];
    }

    public function getApprovalSummary(Tournament $tournament): array
    {
        $completeness = $this->getPlayerCompletenessStatus($tournament);
        $approvalCheck = $this->canBeApprovedForTournament($tournament);
        
        return [
            'team' => [
                'id' => $this->id,
                'name' => $this->name,
                'captain' => $this->captain,
                'status' => $this->getApprovalStatus($tournament),
                'registered_at' => $this->tournaments()->where('tournament_id', $tournament->id)->first()?->pivot->registered_at,
            ],
            'players' => [
                'total' => $completeness['total_players'],
                'complete' => $completeness['complete_players'],
                'incomplete' => count($completeness['incomplete_players']),
                'completeness_percentage' => $completeness['team_completeness_percentage'],
                'details' => $completeness['incomplete_players'],
            ],
            'requirements' => [
                'minimum_players_met' => $completeness['minimum_players_met'],
                'recommended_players_met' => $completeness['recommended_players_met'],
                'all_players_complete' => $completeness['is_team_complete'],
            ],
            'approval' => $approvalCheck,
        ];
    }

    /**
     * Get category display name
     */
    public function getCategoryName(): string
    {
        $categories = config('tournament.categories', []);
        return $categories[$this->category]['name'] ?? ucfirst(str_replace('_', ' ', $this->category ?? 'men'));
    }

    /**
     * Get category configuration
     */
    public function getCategoryConfig(): array
    {
        $categories = config('tournament.categories', []);
        return $categories[$this->category ?? 'men'] ?? [];
    }

    /**
     * Validate if team's players match category requirements
     */
    public function validatePlayersMatchCategory(?Tournament $tournament = null): array
    {
        $validationService = new \App\Services\CategoryValidationService();
        $tournamentStartDate = $tournament ? $tournament->start_date : null;
        
        return $validationService->validateTeamCategory($this, $tournamentStartDate);
    }

    /**
     * Check if team is eligible for a specific category
     */
    public function isCategoryEligible(string $category, ?Tournament $tournament = null): bool
    {
        $this->category = $category;
        $validation = $this->validatePlayersMatchCategory($tournament);
        return $validation['valid'];
    }
}

