<?php

namespace App\Models;

use Illuminate\Database\Eloquent\Factories\HasFactory;
use Illuminate\Database\Eloquent\Model;
use Illuminate\Database\Eloquent\Relations\BelongsToMany;
use Illuminate\Database\Eloquent\Builder;

class Role extends Model
{
    use HasFactory;

    protected $fillable = [
        'name',
        'display_name',
        'description',
        'permissions',
        'is_system',
        'is_active',
        'priority',
        'color',
        'icon',
        'metadata',
    ];

    protected $casts = [
        'permissions' => 'array',
        'is_system' => 'boolean',
        'is_active' => 'boolean',
        'metadata' => 'array',
    ];

    // Relationships
    public function users(): BelongsToMany
    {
        return $this->belongsToMany(User::class)
                    ->withPivot(['is_primary', 'assigned_at', 'expires_at', 'metadata'])
                    ->withTimestamps();
    }

    public function permissions()
    {
        return $this->belongsToMany(Permission::class, 'permission_role')
                    ->withTimestamps();
    }

    // Scopes
    public function scopeActive(Builder $query): void
    {
        $query->where('is_active', true);
    }

    public function scopeSystem(Builder $query): void
    {
        $query->where('is_system', true);
    }

    public function scopeCustom(Builder $query): void
    {
        $query->where('is_system', false);
    }

    public function scopeByPriority(Builder $query): void
    {
        $query->orderBy('priority', 'desc');
    }

    // Accessors
    public function getIsExpiredAttribute(): bool
    {
        return $this->pivot && $this->pivot->expires_at && $this->pivot->expires_at->isPast();
    }

    public function getIsPrimaryAttribute(): bool
    {
        return $this->pivot && $this->pivot->is_primary;
    }

    // Methods
    public function hasPermission(string $permission): bool
    {
        // Check for super admin wildcard permission
        if ($this->permissions && in_array('*', $this->permissions)) {
            return true;
        }
        
        // Check legacy JSON permissions first
        if ($this->permissions && is_array($this->permissions)) {
            if (in_array($permission, $this->permissions)) {
                return true;
            }
        }
        
        // Check new Permission model relationships
        $permissionRecord = $this->permissions()
            ->where('name', $permission)
            ->where('is_active', true)
            ->first();
            
        return $permissionRecord !== null;
    }

    public function hasAnyPermission(array $permissions): bool
    {
        foreach ($permissions as $permission) {
            if ($this->hasPermission($permission)) {
                return true;
            }
        }
        return false;
    }

    public function hasAllPermissions(array $permissions): bool
    {
        foreach ($permissions as $permission) {
            if (!$this->hasPermission($permission)) {
                return false;
            }
        }
        return true;
    }

    public function addPermission(string $permission): void
    {
        $permissions = $this->permissions ?? [];
        if (!in_array($permission, $permissions)) {
            $permissions[] = $permission;
            $this->update(['permissions' => $permissions]);
        }
    }

    public function removePermission(string $permission): void
    {
        $permissions = $this->permissions ?? [];
        $permissions = array_filter($permissions, fn($p) => $p !== $permission);
        $this->update(['permissions' => array_values($permissions)]);
    }

    

    public function assignToUser(User $user, bool $isPrimary = false): void
    {
        $this->users()->attach($user->id, [
            'is_primary' => $isPrimary,
            'assigned_at' => now(),
        ]);
    }

    public function removeFromUser(User $user): void
    {
        $this->users()->detach($user->id);
    }

    public function isSystemRole(): bool
    {
        return $this->is_system;
    }

    public function isActive(): bool
    {
        return $this->is_active;
    }

    public function deactivate(): void
    {
        $this->update(['is_active' => false]);
    }

    public function activate(): void
    {
        $this->update(['is_active' => true]);
    }

    public function getBadgeColor(): string
    {
        return $this->color ?? 'bg-gray-500';
    }

    // Permission management methods
    public function assignPermission($permission): void
    {
        if (is_string($permission)) {
            $permission = Permission::where('name', $permission)->first();
        }
        
        if ($permission) {
            $this->permissions()->syncWithoutDetaching([$permission->id]);
        }
    }


    public function syncPermissions(array $permissions): void
    {
        if (is_array($permissions) && count($permissions) > 0 && is_string($permissions[0])) {
            // If array of permission names, convert to IDs
            $permissionIds = Permission::whereIn('name', $permissions)->pluck('id')->toArray();
        } else {
            // If array of Permission models or IDs
            $permissionIds = collect($permissions)->map(function ($permission) {
                return is_object($permission) ? $permission->id : $permission;
            })->toArray();
        }
        
        $this->permissions()->sync($permissionIds);
    }

    public function getAllPermissions(): \Illuminate\Support\Collection
    {
        // Get legacy JSON permissions
        $legacyPermissions = collect($this->permissions ?? []);
        
        // Get new Permission model permissions
        $modelPermissions = $this->permissions()->where('is_active', true)->pluck('name');
        
        return $legacyPermissions->concat($modelPermissions)->unique();
    }

    public function getIcon(): string
    {
        return $this->icon ?? 'shield';
    }
}
